<?php

declare(strict_types=1);

namespace Drupal\meta_pixel;

use Drupal\Core\Cache\RefinableCacheableDependencyInterface;
use Drupal\Core\Cache\RefinableCacheableDependencyTrait;
use Drupal\Core\Entity\EntityInterface;

/**
 * Represents a Meta Pixel tracking event ready for transmission.
 *
 * This is a value object that holds the final event data after plugin
 * processing and alter hooks. It's used to transmit events to both the
 * browser pixel (via JavaScript) and the Conversions API (server-side).
 *
 * The class implements RefinableCacheableDependencyInterface to properly
 * track cache metadata from the entities and data used to build the event.
 */
class MetaPixelEvent implements RefinableCacheableDependencyInterface {

  use RefinableCacheableDependencyTrait;

  /**
   * The Meta event name.
   *
   * Standard Meta event names like 'PageView', 'ViewContent', 'AddToCart',
   * 'Purchase', etc.
   *
   * @var string
   */
  protected string $eventName;

  /**
   * The event-specific parameters.
   *
   * Contains Meta-formatted event data such as content_ids, value, currency,
   * contents, etc. Does NOT include event_name or event_id as those are
   * stored separately.
   *
   * @var array
   */
  protected array $eventData;

  /**
   * The user identification data.
   *
   * Contains user data for advanced matching and CAPI, with structure:
   * - profiles: Array of identity sets keyed by source (billing, shipping).
   * - external_ids: Array of external identifiers.
   * - emails: Array of email addresses.
   * - phones: Array of phone numbers.
   *
   * @var array
   */
  protected array $userData;

  /**
   * The unique event identifier.
   *
   * Used for deduplication between browser pixel and server-side CAPI
   * to prevent Meta from counting the same event twice.
   *
   * @var string
   */
  protected string $eventId;

  /**
   * The Unix timestamp of the event.
   *
   * @var int
   */
  protected int $eventTime;

  /**
   * Constructs a MetaPixelEvent object.
   *
   * @param string $event_name
   *   The Meta event name (e.g., 'AddToCart', 'Purchase').
   * @param array $event_data
   *   The event-specific parameters. May contain 'event_id' which will be
   *   extracted, and 'event_name' which will be removed to avoid duplication.
   * @param array $user_data
   *   The user identification data for advanced matching.
   */
  public function __construct(string $event_name, array $event_data, array $user_data) {
    $this->eventName = $event_name;
    $this->eventId = $event_data['event_id'] ?? uniqid();
    $this->eventTime = $event_data['event_time'] ?? time();

    // Some values are passed through the event data so they can be altered
    // via hook_meta_pixel_event_data_alter(), but they shouldn't be in the
    // final array since they're stored as separate properties.
    unset($event_data['event_name']);
    unset($event_data['event_id']);
    unset($event_data['event_time']);

    $this->eventData = $event_data;
    $this->userData = $user_data;
  }

  /**
   * Gets the Meta event name.
   *
   * @return string
   *   The event name (e.g., 'ViewContent', 'AddToCart').
   */
  public function getName(): string {
    return $this->eventName;
  }

  /**
   * Gets the unique event identifier.
   *
   * @return string
   *   The event ID used for deduplication between pixel and CAPI.
   */
  public function getEventId(): string {
    return $this->eventId ?? '';
  }

  /**
   * Gets the Unix timestamp of the event.
   *
   * @return int
   *   The event time.
   */
  public function getEventTime(): int {
    return $this->eventTime;
  }

  /**
   * Gets the event-specific parameters.
   *
   * Returns the Meta-formatted event data without event_name or event_id,
   * which are accessed via separate getters.
   *
   * @return array
   *   The event data array (content_ids, value, currency, etc.).
   */
  public function getEventData(): array {
    return $this->eventData;
  }

  /**
   * Gets the user identification data.
   *
   * @return array
   *   The user data array for advanced matching and CAPI.
   */
  public function getUserData(): array {
    return $this->userData;
  }

  /**
   * Sets/updates the event-specific parameters.
   *
   * Allows modification of event data after construction, typically used
   * by EventCollector after applying alter hooks.
   *
   * @param array $data
   *   The updated event data array.
   */
  public function setEventData(array $data): void {
    foreach ($data as $key => $value) {
      // We don't want to store any full entities in the final event.
      if ($value instanceof EntityInterface) {
        unset($data[$key]);
      }
    }
    $this->eventData = $data;
  }

}
