<?php

declare(strict_types=1);

namespace Drupal\meta_pixel;

use Drupal\Component\Plugin\PluginBase;
use Drupal\Component\Utility\NestedArray;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Base class for Meta Pixel event plugins.
 *
 * Provides a stateful plugin pattern where plugins can be initialized with
 * runtime data and then queried via getters. Plugins are cloned to avoid
 * mutating cached instances.
 */
abstract class MetaPixelEventPluginBase extends PluginBase implements MetaPixelEventPluginInterface, ContainerFactoryPluginInterface {

  use StringTranslationTrait;

  /**
   * The context data for this event instance.
   *
   * Contains entities and other contextual information needed to build
   * event data (e.g., product, order, user entities).
   *
   * @var array
   */
  protected array $data;

  /**
   * The generated unique event ID.
   *
   * Used for deduplication between browser pixel and CAPI.
   *
   * @var string
   */
  protected string $id;

  /**
   * Optional override for the event name.
   *
   * Allows runtime override of the event name defined in plugin definition.
   *
   * @var string
   */
  protected string $name;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition
    );
  }

  /**
   * Creates a new plugin instance initialized with context data.
   *
   * Clones the plugin to avoid mutating the cached singleton instance,
   * then initializes the clone with the provided data, generates an event ID,
   * and returns the initialized instance.
   *
   * @param array $data
   *   Context data containing entities and other information needed to build
   *   the event. May include an 'event_name' key to override the default.
   *
   * @return static
   *   A cloned and initialized plugin instance.
   */
  public function createInstance(array $data = []) {
    // Clone to avoid mutating the cached plugin instance.
    $instance = clone $this;

    if (!empty($data['event_name'])) {
      $instance->name = $data['event_name'];
    }
    unset($data['event_name']);

    $instance->data = $data;
    $instance->id = $instance->generateEventId();

    return $instance;
  }

  /**
   * Sets the context data for this event instance.
   *
   * @param array $data
   *   The context data array.
   *
   * @return $this
   */
  public function setData(array $data = []) {
    $this->data = $data;
    return $this;
  }

  /**
   * Gets the context data for this event instance.
   *
   * @return array
   *   The context data array, or empty array if not set.
   */
  public function getData() {
    return $this->data ?? [];
  }

  /**
   * {@inheritdoc}
   */
  public function getEventName(): string {
    return $this->name ?? $this->pluginDefinition['event_name'];
  }

  /**
   * {@inheritdoc}
   */
  public function getLabel(): string {
    return (string) $this->pluginDefinition['label'];
  }

  /**
   * {@inheritdoc}
   */
  public function getDescription(): string {
    return (string) $this->pluginDefinition['description'];
  }

  /**
   * {@inheritdoc}
   */
  public function getWeight(): int {
    return $this->pluginDefinition['weight'] ?? 0;
  }

  /**
   * {@inheritdoc}
   */
  public function applies(array $data): bool {
    // Default implementation - subclasses should override.
    return TRUE;
  }

  /**
   * Generates a unique event ID for deduplication.
   *
   * Creates an event ID using the event name and any entity IDs found in
   * the context data, plus a unique suffix. This ensures browser pixel and
   * CAPI use the same ID for proper deduplication.
   *
   * Format: {event_name}-{entity_id}-{entity_id}-{uniqid}
   * Example: ViewContent-123-456-507f1f77bcf86cd799439011
   *
   * @return string
   *   The generated event ID.
   */
  protected function generateEventId(): string {
    $data = $this->getData();
    $ids = [];

    // Extract entity IDs from context data.
    foreach ($data as $key => $entity) {
      if ($entity instanceof ContentEntityInterface) {
        $ids[] = $entity->id();
      }
    }

    // Add unique identifier.
    $ids[] = uniqid();

    return $this->getEventName() . '-' . implode('-', $ids);
  }

  /**
   * Gets the generated event ID.
   *
   * @return string
   *   The unique event ID for deduplication.
   */
  public function getEventId(): string {
    return $this->id;
  }

  /**
   * Builds the base event data structure.
   *
   * Child classes should call parent::buildEventData() and merge their
   * own event-specific data into the returned array.
   *
   * @return array
   *   Base event data containing:
   *   - event_name: The Meta event name.
   *   - event_id: The unique event ID for deduplication.
   *   - event_time: Unix timestamp of when the event occurred.
   */
  public function buildEventData(): array {
    return [
      'event_name' => $this->getEventName(),
      'event_id' => $this->getEventId(),
      'event_time' => time(),
    ];
  }

  /**
   * Converts camelCase to snake_case.
   *
   * Helper method for formatting event names and parameter names.
   *
   * @param string $string
   *   The camelCase string.
   *
   * @return string
   *   The snake_case string.
   */
  protected function camelToSnake(string $string): string {
    return strtolower(preg_replace('/(?<!^)[A-Z]/', '_$0', $string));
  }

  /**
   * Gets the default configuration for this plugin.
   *
   * Provides default values for plugin-specific configuration. Plugins can
   * override this to define their own configurable settings that affect how
   * event data is generated.
   *
   * Example use cases:
   * - Custom parameter mappings
   * - Conditional data inclusion rules
   * - Event-specific formatting options
   *
   * @return array
   *   Default configuration array.
   */
  public function defaultConfiguration(): array {
    return [];
  }

  /**
   * Gets the current configuration for this plugin.
   *
   * @return array
   *   The current configuration array.
   */
  public function getConfiguration(): array {
    return $this->configuration;
  }

  /**
   * Sets the configuration for this plugin.
   *
   * Merges provided configuration with defaults. Configuration is typically
   * set from admin forms and can affect how buildEventData() generates output.
   *
   * @param array $configuration
   *   The configuration array to set.
   */
  public function setConfiguration(array $configuration) {
    $this->configuration = NestedArray::mergeDeep($this->defaultConfiguration(), $configuration);
  }

  /**
   * Builds the configuration form for this plugin.
   *
   * Plugins can override this to provide admin UI for configuring
   * event-specific behavior. The form will be displayed in the event
   * configuration page.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array
   *   The form array with plugin-specific configuration fields.
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    return $form;
  }

  /**
   * Validates the configuration form for this plugin.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state): void {
    // No validation by default.
  }

  /**
   * Handles submission of the configuration form for this plugin.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    // No submission handling by default.
  }

}
