<?php

declare(strict_types=1);

namespace Drupal\meta_pixel;

use Drupal\Component\Plugin\PluginInspectionInterface;

/**
 * Interface for Meta Pixel event plugins.
 *
 * Event plugins use a stateful pattern where they are initialized with
 * context data and then queried via getters. Plugins build event data
 * structures that can be sent to both browser pixel and Conversions API.
 */
interface MetaPixelEventPluginInterface extends PluginInspectionInterface {

  /**
   * Creates a new plugin instance initialized with context data.
   *
   * This method clones the plugin and initializes the clone with the provided
   * context data, allowing the same plugin definition to be used for multiple
   * events without state conflicts.
   *
   * @param array $data
   *   Context data containing entities and other information needed to build
   *   the event. Common keys include:
   *   - product: Product entity.
   *   - product_variation: Product variation entity.
   *   - order: Order entity.
   *   - item: Order item entity.
   *   - node: Node entity.
   *   - user: User entity.
   *   - event_name: Optional override for the event name.
   *
   * @return static
   *   A cloned and initialized plugin instance.
   */
  public function createInstance(array $data = []);

  /**
   * Gets the Meta event name.
   *
   * Returns the standard Meta event name as defined in the plugin definition,
   * or an overridden name if set during instance creation.
   *
   * @return string
   *   The Meta event name (e.g., 'ViewContent', 'AddToCart', 'Purchase').
   */
  public function getEventName(): string;

  /**
   * Gets the human-readable event label.
   *
   * @return string
   *   The translatable label for admin UI display.
   */
  public function getLabel(): string;

  /**
   * Gets the event description.
   *
   * @return string
   *   A brief description of when this event fires.
   */
  public function getDescription(): string;

  /**
   * Gets the plugin weight for sorting.
   *
   * Used when multiple plugins claim the same event name. Lower weights
   * are processed first.
   *
   * @return int
   *   The weight (default 0).
   */
  public function getWeight(): int;

  /**
   * Checks if this plugin can handle the given context data.
   *
   * Validates that required entities or data are present before attempting
   * to build event data. For example, a ViewContent plugin for products
   * would check for a product_variation entity.
   *
   * @param array $data
   *   Context data that may contain entities and other information.
   *
   * @return bool
   *   TRUE if the plugin can build event data from this context, FALSE
   *   otherwise.
   */
  public function applies(array $data): bool;

  /**
   * Builds Meta-formatted event data from the initialized context.
   *
   * Transforms Drupal entities into the data structure required by Meta
   * Pixel and Conversions API. This method is called after the plugin
   * instance has been initialized with context data via createInstance().
   *
   * Child implementations should call parent::buildEventData() to get the
   * base structure (event_name, event_id, event_time) and merge their
   * event-specific data.
   *
   * @return array
   *   Meta-formatted event data. Base structure includes:
   *   - event_name: The Meta event name.
   *   - event_id: Unique ID for deduplication.
   *   - event_time: Unix timestamp.
   *   Event-specific data may include:
   *   - content_ids: Array of product IDs/SKUs.
   *   - content_type: Type of content (e.g., 'product', 'node').
   *   - content_name: Name/title of the content.
   *   - value: Monetary value for commerce events.
   *   - currency: Currency code (e.g., 'USD').
   *   - contents: Array of detailed item information.
   *   - num_items: Number of items for commerce events.
   *   - order_id: Order identifier for purchase events.
   */
  public function buildEventData(): array;

  /**
   * Gets the generated unique event ID.
   *
   * Returns the event ID that was generated during instance initialization.
   * This ID is used for deduplication between browser pixel and server-side
   * CAPI to prevent Meta from double-counting events.
   *
   * @return string
   *   The unique event ID (e.g., 'ViewContent-123-456-507f1f77bcf86cd39011').
   */
  public function getEventId(): string;

  /**
   * Gets the context data for this event instance.
   *
   * Returns the data that was provided during instance initialization,
   * containing entities and other contextual information.
   *
   * @return array
   *   The context data array.
   */
  public function getData();

}
