<?php

declare(strict_types=1);

namespace Drupal\meta_pixel\Plugin\Block;

use Drupal\Core\Block\BlockBase;
use Drupal\Core\Cache\Cache;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\meta_pixel\EventCollector;
use Drupal\meta_pixel\Plugin\MetaPixelEventPluginManager;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\Yaml\Yaml;

/**
 * Provides a Meta Pixel Event Emitter block.
 *
 * This block allows site builders to trigger Meta Pixel events from specific
 * block placements. Useful for tracking micro-conversions, content engagement,
 * or custom events tied to specific page regions or layouts.
 *
 * @Block(
 *   id = "meta_pixel_emitter",
 *   admin_label = @Translation("Meta Pixel Event Emitter"),
 *   category = @Translation("Marketing"),
 * )
 */
class MetaPixelEmitterBlock extends BlockBase implements ContainerFactoryPluginInterface {

  /**
   * The Meta Pixel event collector service.
   *
   * @var \Drupal\meta_pixel\EventCollector
   */
  protected EventCollector $eventCollector;

  /**
   * The Meta Pixel event plugin manager.
   *
   * @var \Drupal\meta_pixel\Plugin\MetaPixelEventPluginManager
   */
  protected MetaPixelEventPluginManager $eventPluginManager;

  /**
   * Constructs a MetaPixelEmitterBlock object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin ID for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\meta_pixel\EventCollector $event_collector
   *   The event collector service.
   * @param \Drupal\meta_pixel\Plugin\MetaPixelEventPluginManager $event_plugin_manager
   *   The event plugin manager.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    EventCollector $event_collector,
    MetaPixelEventPluginManager $event_plugin_manager,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->eventCollector = $event_collector;
    $this->eventPluginManager = $event_plugin_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('meta_pixel.event_collector'),
      $container->get('meta_pixel.event_plugin_manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'event_plugin_id' => '',
      'event_name' => '',
      'custom_properties' => '',
      'debug' => FALSE,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function blockForm($form, FormStateInterface $form_state) {
    $form = parent::blockForm($form, $form_state);

    // Get available event plugins.
    $plugin_definitions = $this->eventPluginManager->getDefinitions();
    $event_options = ['' => $this->t('- Select event -')];

    foreach ($plugin_definitions as $plugin_id => $definition) {
      $event_options[$plugin_id] = $definition['label'] . ' (' . $definition['event_name'] . ')';
    }

    // Page view is a unique case. To improve caching, we include this in the
    // main JS so a lazy builder doesn't need to be used. When lazy builders
    // are present the full page cache is bypassed and Drupal uses the dynamic
    // page cache. If the only event on a page is page view there is no need
    // to use the lazy builder. We don't want people to mistakenly use it in a
    // block.
    unset($event_options['page_view']);

    $form['event_plugin_id'] = [
      '#type' => 'select',
      '#title' => $this->t('Event Type'),
      '#description' => $this->t('Select which Meta Pixel event to fire when this block is displayed.'),
      '#options' => $event_options,
      '#default_value' => $this->configuration['event_plugin_id'],
      '#required' => TRUE,
    ];

    $form['event_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Custom Event Name'),
      '#description' => $this->t('Optional: Override the event name. Leave empty to use the default event name from the selected plugin. Only use this for custom events or special tracking needs.'),
      '#default_value' => $this->configuration['event_name'],
      '#maxlength' => 255,
    ];

    $form['custom_properties'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Custom Event Properties'),
      '#description' => $this->t('Optional: Add custom event properties in YAML format. Example:<br><code>content_name: Special Offer<br>custom_field: custom_value</code>'),
      '#default_value' => $this->configuration['custom_properties'],
      '#attributes' => ['data-yaml-editor' => TRUE],
      '#rows' => 5,
    ];

    $form['debug'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Display debug information'),
      '#description' => $this->t('Show event data in YAML format on the page. <strong>Warning:</strong> Only enable this during development. Do not leave enabled on production sites.'),
      '#default_value' => $this->configuration['debug'],
    ];

    $form['visibility_note'] = [
      '#type' => 'markup',
      '#markup' => '<p><strong>' . $this->t('Note:') . '</strong> ' . $this->t('Events will only fire if tracking is enabled for the selected event type in the Meta Pixel settings. Additionally, user role and page visibility rules will still apply.') . '</p>',
      '#weight' => 100,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function blockValidate($form, FormStateInterface $form_state) {
    parent::blockValidate($form, $form_state);

    $custom_properties = $form_state->getValue('custom_properties');

    if (!empty($custom_properties)) {
      try {
        // Validate YAML syntax.
        $parsed = Yaml::parse($custom_properties);

        if (!is_array($parsed)) {
          $form_state->setErrorByName('custom_properties', $this->t('Custom properties must be valid YAML key-value pairs.'));
        }
      }
      catch (\Exception $e) {
        $form_state->setErrorByName('custom_properties', $this->t('Invalid YAML syntax: @error', [
          '@error' => $e->getMessage(),
        ]));
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function blockSubmit($form, FormStateInterface $form_state) {
    parent::blockSubmit($form, $form_state);

    $this->configuration['event_plugin_id'] = $form_state->getValue('event_plugin_id');
    $this->configuration['event_name'] = $form_state->getValue('event_name');
    $this->configuration['custom_properties'] = $form_state->getValue('custom_properties');
    $this->configuration['debug'] = $form_state->getValue('debug');
  }

  /**
   * {@inheritdoc}
   */
  public function build() {
    $plugin_id = $this->configuration['event_plugin_id'];

    if (empty($plugin_id)) {
      return [];
    }

    $build['meta_pixel_emitter'] = [
      '#lazy_builder' => [
        'meta_pixel.lazy_builder:emitEvent',
        [
          $plugin_id,
          $this->configuration['event_name'] ?? '',
          $this->configuration['custom_properties'] ?? '',
          $this->configuration['debug'] ? '1' : '0',
        ],
      ],
      '#create_placeholder' => TRUE,
      '#cache' => [
        'keys' => [
          'meta_pixel_emitter',
          $plugin_id,
        ],
      ],
    ];
    return $build;
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheMaxAge() {
    return Cache::PERMANENT;
  }

}
