<?php

declare(strict_types=1);

namespace Drupal\meta_pixel\Plugin;

use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Plugin\DefaultPluginManager;
use Drupal\meta_pixel\MetaPixelEventPluginInterface;

/**
 * Manages discovery and instantiation of Meta Pixel event plugins.
 *
 * @see \Drupal\meta_pixel\Annotation\MetaPixelEvent
 * @see \Drupal\meta_pixel\MetaPixelEventPluginInterface
 * @see \Drupal\meta_pixel\MetaPixelEventBase
 * @see plugin_api
 */
class MetaPixelEventPluginManager extends DefaultPluginManager {

  /**
   * Constructs a MetaPixelEventPluginManager object.
   *
   * @param \Traversable $namespaces
   *   An object that implements \Traversable which contains the root paths
   *   keyed by the corresponding namespace to look for plugin implementations.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache_backend
   *   Cache backend instance to use.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler to invoke the alter hook with.
   */
  public function __construct(\Traversable $namespaces, CacheBackendInterface $cache_backend, ModuleHandlerInterface $module_handler) {
    parent::__construct(
      'Plugin/MetaPixelEvent',
      $namespaces,
      $module_handler,
      'Drupal\meta_pixel\MetaPixelEventPluginInterface',
      'Drupal\meta_pixel\Attribute\MetaPixelEvent',
      'Drupal\meta_pixel\Annotation\MetaPixelEvent'
    );

    $this->alterInfo('meta_pixel_event_info');
    $this->setCacheBackend($cache_backend, 'meta_pixel_event_plugins');
  }

  /**
   * Gets event plugins for a specific Meta event name.
   *
   * Multiple plugins can handle the same event name, sorted by weight.
   *
   * @param string $event_name
   *   The Meta event name (e.g., 'ViewContent').
   *
   * @return \Drupal\meta_pixel\MetaPixelEventPluginInterface[]
   *   Array of plugin instances, sorted by weight.
   */
  public function getEventPlugins(string $id): array {
    $plugins = [];
    $definitions = $this->getDefinitions();

    foreach ($definitions as $plugin_id => $definition) {
      if ($definition['id'] === $id) {
        $plugins[] = $this->createInstance($plugin_id);
      }
    }

    // Sort by weight.
    usort($plugins, function ($a, $b) {
      return $a->getWeight() <=> $b->getWeight();
    });

    return $plugins;
  }

  /**
   * Gets the first applicable plugin for an event and context.
   *
   * @param string $event_name
   *   The Meta event name.
   * @param array $context
   *   Context array with entities.
   *
   * @return \Drupal\meta_pixel\MetaPixelEventPluginInterface|null
   *   The first applicable plugin, or NULL if none apply.
   */
  public function getApplicablePlugin(string $id, array $context): ?MetaPixelEventPluginInterface {
    $plugins = $this->getEventPlugins($id);

    foreach ($plugins as $plugin) {
      if ($plugin->applies($context)) {
        return $plugin->createInstance($context);
      }
    }

    return NULL;
  }

  /**
   * Gets all available Meta event names from plugin definitions.
   *
   * @return array
   *   Array of event names keyed by plugin ID.
   */
  public function getEventNames(): array {
    $event_names = [];
    $definitions = $this->getDefinitions();

    foreach ($definitions as $plugin_id => $definition) {
      $event_names[$plugin_id] = $definition['event_name'];
    }

    return $event_names;
  }

}
