<?php

declare(strict_types=1);

namespace Drupal\meta_pixel;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Security\TrustedCallbackInterface;

/**
 * Lazy builder for ViewContent event tracking.
 *
 * @todo There are currently two lazy builder classes, they should be
 * consolidated into one. LazyBuilderEmitter should be the one we keep.
 */
class ViewContentLazyBuilder implements TrustedCallbackInterface {

  /**
   * The event collector.
   *
   * @var \Drupal\meta_pixel\EventCollector
   */
  private EventCollector $eventCollector;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  private EntityTypeManagerInterface $entityTypeManager;

  /**
   * Constructs a ViewContentLazyBuilder object.
   *
   * @param \Drupal\meta_pixel\EventCollector $eventCollector
   *   The event collector.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   */
  public function __construct(
    EventCollector $eventCollector,
    EntityTypeManagerInterface $entityTypeManager
  ) {
    $this->eventCollector = $eventCollector;
    $this->entityTypeManager = $entityTypeManager;
  }

  /**
   * {@inheritdoc}
   */
  public static function trustedCallbacks() {
    return ['trackViewContent'];
  }

  /**
   * Lazy builder callback for ViewContent event.
   *
   * This fires regardless of whether the product is in stock or not.
   *
   * @param string $entity_type
   *   The entity type ('commerce_product' or 'node').
   * @param int $entity_id
   *   The entity ID.
   * @param int $variation_id
   *   The product variation ID (0 for nodes).
   *
   * @return array
   *   Empty render array (event is fired, nothing to display).
   */
  public function trackViewContent(string $entity_type, int $entity_id, int|null $variation_id = NULL): array {
    // Cache metadata, we don't want to cache this ever because it needs to
    // execute every time its viewed.
    $build = [
      '#cache' => [
        'max-age' => 0,
      ],
    ];

    if ($entity_type === 'commerce_product') {
      $this->trackProductView($entity_id, $variation_id);
    }
    elseif ($entity_type === 'node') {
      $this->trackNodeView($entity_id);
    }

    // Attach the browser events to the settings variable of this render array
    // so it triggers the js fbq(). CAPI events are collected and sent in a
    // single request in the kernel.terminate event.
    $events = [];
    foreach ($this->eventCollector->getBrowserEvents() as $event) {
      $events[$event->getEventId()] = [
        'event_name' => $event->getName(),
        'event_data' => $event->getEventData(),
        'event_id' => $event->getEventId(),
      ];
    }

    if ($events) {
      $build['#attached']['drupalSettings']['metaPixel']['events'] = $events;
    }

    return $build;
  }

  /**
   * Fires ViewContent event for commerce products.
   *
   * @param int $product_id
   *   The product ID.
   * @param int $variation_id
   *   The variation ID.
   */
  protected function trackProductView(int $product_id, int $variation_id): void {
    /** @var \Drupal\commerce_product\Entity\ProductInterface $product */
    $product = $this->entityTypeManager->getStorage('commerce_product')->load($product_id);
    if (!$product) {
      return;
    }

    /** @var \Drupal\commerce_product\Entity\ProductVariationInterface $variation */
    $variation = $this->entityTypeManager->getStorage('commerce_product_variation')->load($variation_id);
    if (!$variation) {
      return;
    }

    // Add context for caching and alter hook
    $data['product'] = $product;
    $data['product_variation'] = $variation;

    $this->eventCollector->addEvent('commerce_view_content', $data);
  }

  /**
   * Fires ViewContent event for nodes.
   *
   * @param int $node_id
   *   The node ID.
   *
   */
  protected function trackNodeView(int $node_id): void {
    /** @var \Drupal\node\NodeInterface $node */
    $node = $this->entityTypeManager->getStorage('node')->load($node_id);
    if (!$node) {
      return;
    }

    $data['node'] = $node;

    $this->eventCollector->addEvent('node_view_content', $data);
  }

}
