<?php

namespace Drupal\metadata_sanitizer\Commands;

use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\file\FileUsage\FileUsageInterface;
use Drush\Attributes as CLI;
use Drush\Commands\DrushCommands;
use Symfony\Component\Process\Process;

/**
 * Drush commands for Metadata Sanitizer.
 */
class MetadataSanitizerCommands extends DrushCommands {

  /**
   * The entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The file system service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected FileSystemInterface $fileSystem;

  /**
   * The file usage service.
   *
   * @var \Drupal\file\FileUsage\FileUsageInterface
   */
  protected FileUsageInterface $fileUsage;

  /**
   * The file entity storage.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface
   */
  protected EntityStorageInterface $fileStorage;

  /**
   * Constructs a new MetadataSanitizerCommands object.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  public function __construct(
    EntityTypeManagerInterface $entityTypeManager,
    FileSystemInterface $fileSystem,
    FileUsageInterface $fileUsage,
  ) {
    parent::__construct();
    $this->entityTypeManager = $entityTypeManager;
    $this->fileSystem = $fileSystem;
    $this->fileUsage = $fileUsage;
    $this->fileStorage = $entityTypeManager->getStorage('file');
  }

  /**
   * Removes metadata from all files in the system, with optional filters.
   *
   * @param array $options
   *   (optional) An associative array of options to filter which files are
   *     processed.
   *   - extensions: (string|null) Comma-separated list of file extensions to
   *     include.
   *   - pattern: (string|null) Regex pattern to match filenames.
   *   - mime: (string|null) Comma-separated list of MIME types
   *     (e.g. image/jpeg, application/pdf).
   *   - field: (string|null) Restrict to files referenced by a specific entity
   *     field name.
   */
  #[CLI\Command(name: 'metadata_sanitizer:clean', aliases: ['msc'])]
  #[CLI\Help(description: 'Remove metadata from all files in the system, with optional filters.')]
  #[CLI\Option(name: 'extensions', description: 'Comma-separated list of file extensions to include (e.g. jpg,png,pdf).')]
  #[CLI\Option(name: 'pattern', description: 'Regex pattern to match filenames (e.g. /report/i).')]
  #[CLI\Option(name: 'mime', description: 'Comma-separated list of MIME types (e.g. image/jpeg,application/pdf).')]
  #[CLI\Option(name: 'field', description: 'Restrict to files referenced by a specific entity field name.')]
  #[CLI\Usage(name: 'drush metadata_sanitizer:clean --extensions=jpg,pdf', description: 'Remove metadata from all .jpg and .pdf files.')]
  #[CLI\Usage(name: 'drush metadata_sanitizer:clean --pattern="/^invoice_/"', description: "Remove metadata from files whose names start with 'invoice_'.")]
  #[CLI\Usage(name: 'drush metadata_sanitizer:clean --mime="image/png"', description: 'Remove metadata from PNG images only.')]
  #[CLI\Usage(name: 'drush metadata_sanitizer:clean --field=field_document', description: "Remove metadata only from files referenced by the 'field_document' field.")]
  public function clean(
    array $options = [
      'extensions' => NULL,
      'pattern' => NULL,
      'mime' => NULL,
      'field' => NULL,
    ],
  ): void {
    $this->output()->writeln('<info>Starting metadata cleanup...</info>');
    $fids = $this->entityTypeManager->getStorage('file')->getQuery()->accessCheck(FALSE)->execute();
    $processed = 0;
    foreach ($fids as $fid) {
      $file = $this->fileStorage->load($fid);
      if (!$file) {
        continue;
      }
      $uri = $file->getFileUri();
      $path = $this->fileSystem->realpath($uri);
      if (!$path) {
        continue;
      }

      if (!$this->shouldProcessFile($file, $path, $fid, $options)) {
        continue;
      }

      $this->sanitizeFile($path);
      $this->output()->writeln("<comment>Cleaned metadata on: $path</comment>");
      $processed++;
    }
    $this->output()->writeln("<info>Metadata cleanup completed. Total files processed: $processed</info>");
  }

  /**
   * Checks if a file matches all filter options and should be processed.
   *
   * @param \Drupal\file\FileInterface $file
   *   The file entity.
   * @param string $path
   *   The file system path.
   * @param int $fid
   *   The file ID.
   * @param array $options
   *   Filter options (extension, pattern, mime, field).
   *
   * @return bool
   *   TRUE if the file passes all filters, FALSE otherwise.
   */
  protected function shouldProcessFile($file, $path, $fid, array $options): bool {

    // Filter by extensions.
    if (!empty($options['extensions'])) {
      $exts = array_map('trim', explode(',', $options['extensions']));
      $ext = mb_strtolower(pathinfo($path, PATHINFO_EXTENSION));
      if (!in_array($ext, $exts, TRUE)) {
        return FALSE;
      }
    }

    // Filter by filename pattern.
    if (!empty($options['pattern']) && !preg_match($options['pattern'], $this->fileSystem->basename($path))) {
      return FALSE;
    }

    // Filter by MIME type.
    if (!empty($options['mime'])) {
      $mimes = array_map('trim', explode(',', $options['mime']));
      if (!in_array($file->getMimeType(), $mimes, TRUE)) {
        return FALSE;
      }
    }

    // Filter by field usage.
    if (!empty($options['field'])) {
      $usage = $this->fileUsage->listUsage($file);
      $found = FALSE;
      foreach ($usage as $entities) {
        foreach ($entities as $entity_type => $items) {
          foreach ($items as $entity_id) {
            $entity = $this->entityTypeManager->getStorage($entity_type)->load($entity_id);
            if ($entity && $entity->hasField($options['field'])) {
              $field = $entity->get($options['field']);
              if ($field->isEmpty()) {
                continue;
              }
              foreach ($field->getValue() as $item) {
                if (!empty($item['target_id']) && $item['target_id'] == $fid) {
                  $found = TRUE;
                  break 4;
                }
              }
            }
          }
        }
      }
      if (!$found) {
        return FALSE;
      }
    }
    return TRUE;
  }

  /**
   * Executes exiftool to remove metadata from a file.
   */
  protected function sanitizeFile(string $path): void {
    $process = new Process(['exiftool', '-overwrite_original', '-all=', $path]);
    $process->run();
  }

}
