<?php

namespace Drupal\metadata_sanitizer\Hooks;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\file\FileInterface;
use Psr\Log\LoggerInterface;
use Symfony\Component\Process\Process;

/**
 * Object-oriented hook implementations for metadata_sanitizer.
 */
final class MetadataSanitizerHooks {

  public function __construct(
    private readonly ConfigFactoryInterface $configFactory,
    private readonly FileSystemInterface $fileSystem,
    private readonly LoggerInterface $logger,
  ) {}

  /**
   * Implements hook_file_insert().
   */
  #[Hook('file_insert')]
  public function fileInsert(FileInterface $file): void {
    $config = $this->configFactory->get('metadata_sanitizer.settings');
    if (!$config->get('enabled')) {
      return;
    }

    $extensions = (string) ($config->get('extensions') ?? '');
    $exts = array_filter(array_map('trim', explode(',', $extensions)));

    $uri = $file->getFileUri();
    $path = $this->fileSystem->realpath($uri);
    if (!$path) {
      return;
    }

    if ($exts) {
      $ext = mb_strtolower(pathinfo($path, PATHINFO_EXTENSION));
      if (!in_array($ext, array_map('mb_strtolower', $exts), TRUE)) {
        return;
      }
    }

    $cmd = ['exiftool', '-overwrite_original', '-all='];
    if ($config->get('preserve_dates')) {
      $cmd[] = '-P';
    }
    $cmd[] = $path;

    try {
      $process = new Process($cmd);
      $process->run();
      if (!$process->isSuccessful()) {
        throw new \RuntimeException($process->getErrorOutput() ?: $process->getOutput());
      }
    }
    catch (\Throwable $e) {
      $this->logger->error('Failed to strip metadata from %file: %error', [
        '%file' => $path,
        '%error' => $e->getMessage(),
      ]);
    }
  }

  /**
   * Implements hook_help().
   */
  #[Hook('help')]
  public function help(string $route_name, RouteMatchInterface $route_match): ?array {
    if ($route_name === 'help.page.metadata_sanitizer') {
      return [
        '#markup' => t('Strips metadata from uploaded files using exiftool. Configure at Configuration → Media → Metadata Sanitizer.'),
      ];
    }
    return NULL;
  }

}
