<?php

namespace Drupal\Tests\metatag_page_heading\Functional;

use Drupal\Core\Field\FieldStorageDefinitionInterface;
use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\Tests\BrowserTestBase;
use Drupal\Tests\field_ui\Traits\FieldUiTestTrait;
use Drupal\Tests\taxonomy\Traits\TaxonomyTestTrait;

/**
 * This class provides methods specifically for testing something.
 *
 * @group metatag_page_heading
 */
class MetatagPageHeadingFunctionalTest extends BrowserTestBase {
  use FieldUiTestTrait, TaxonomyTestTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'test_page_test',
    'node',
    'user',
    'taxonomy',
    'field_ui',
    'metatag',
    'metatag_page_heading',
  ];

  /**
   * A user with authenticated permissions.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected $user;

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->config('system.site')->set('page.front', '/test-page')->save();
    $this->user = $this->drupalCreateUser([]);
    $this->drupalLogin($this->rootUser);
  }

  /**
   * Test page heading metatag using the "Content Metatag" default.
   */
  public function testPageHeadingOnNodePageThroughContentMetatagDefault() {
    $session = $this->assertSession();
    $page = $this->getSession()->getPage();

    $this->createContentType(['type' => 'article']);

    // Add page heading metatag content:
    $this->drupalGet('/admin/config/search/metatag/node');
    $page->fillField('page_heading', 'Overwritten name');
    $page->pressButton('edit-submit');
    $session->statusCodeEquals(200);
    $session->pageTextContains('Saved the Content Metatag defaults.');

    // Create a article node:
    $node = $this->createNode([
      'type' => 'article',
      'title' => 'Not overwritten title',
    ]);

    // Go to the article and check the page heading:
    $this->drupalGet($node->toUrl()->toString());
    $session->statusCodeEquals(200);

    $session->elementTextEquals('css', 'h1', 'Overwritten name');

    // Check that the metatag is not present in the HTML head:
    $session->elementNotExists('css', 'meta[name="page_heading"]');
  }

  /**
   * Test page heading metatag using the metatag field.
   */
  public function testPageHeadingOnNodePageThroughMetatagField() {
    $session = $this->assertSession();
    $page = $this->getSession()->getPage();

    $this->createContentType(['type' => 'article']);

    $this->fieldUIAddNewField('/admin/structure/types/manage/article', 'metatags', 'Metatags', 'metatag');

    // Create a article node, with a page heading metatag through the ui
    // (because programmatically it doesn't properly work):
    $this->drupalGet('/node/add/article');
    $page->fillField('title[0][value]', 'Not overwritten title');
    $page->fillField('field_metatags[0][page_heading][page_heading]', 'Overwritten name');
    $page->pressButton('edit-submit');

    $session->statusCodeEquals(200);

    // Go to the article and check the page heading:
    $this->drupalGet('/node/1');
    $session->statusCodeEquals(200);

    $session->elementTextEquals('css', 'h1', 'Overwritten name');

    // Check that the metatag is not present in the HTML head:
    $session->elementNotExists('css', 'meta[name="page_heading"]');

    // Check, that the page heading is only overwritten for that node:
    $node = $this->createNode([
      'type' => 'article',
      'title' => 'Not overwritten title',
    ]);
    $this->drupalGet($node->toUrl()->toString());
    $session->statusCodeEquals(200);
    $session->elementTextEquals('css', 'h1', 'Not overwritten title');

    // Check that the metatag is not present in the HTML head:
    $session->elementNotExists('css', 'meta[name="page_heading"]');
  }

  /**
   * Test page heading metatag using the "User Metatag" default.
   */
  public function testPageHeadingOnUserPageThroughUserMetatagDefault() {
    $session = $this->assertSession();
    $page = $this->getSession()->getPage();

    // Add page heading metatag content:
    $this->drupalGet('/admin/config/search/metatag/user');
    $page->fillField('page_heading', 'Overwritten name');
    $page->pressButton('edit-submit');
    $session->statusCodeEquals(200);
    $session->pageTextContains('Saved the User Metatag defaults.');

    // Create a new user:
    $newUser = $this->createUser([], 'My new User');

    // Go to the user and check the page heading:
    $this->drupalGet($newUser->toUrl()->toString());
    $session->statusCodeEquals(200);

    $session->elementTextEquals('css', 'h1', 'Overwritten name');

    // Check that the metatag is not present in the HTML head:
    $session->elementNotExists('css', 'meta[name="page_heading"]');
  }

  /**
   * Test page heading metatag using the metatag field.
   */
  public function testPageHeadingOnUserPageThroughMetatagField() {
    $session = $this->assertSession();
    $page = $this->getSession()->getPage();
    $this->fieldUIAddNewField('/admin/config/people/accounts', 'metatags', 'Metatags', 'metatag');

    // Create a user node, with a page heading metatag through the ui
    // (because programmatically it doesn't properly work):
    $this->drupalGet('/admin/people/create');
    $page->fillField('name', 'My new User');
    $page->fillField('pass[pass1]', 'password');
    $page->fillField('pass[pass2]', 'password');
    $page->fillField('field_metatags[0][page_heading][page_heading]', 'Overwritten name');
    $page->pressButton('edit-submit');

    $this->assertSession()->statusCodeEquals(200);

    // Go to the user and check the page heading:
    $this->drupalGet('/user/3');
    $session->statusCodeEquals(200);

    $session->elementTextEquals('css', 'h1', 'Overwritten name');

    // Check that the metatag is not present in the HTML head:
    $session->elementNotExists('css', 'meta[name="page_heading"]');

    // Check, that the page heading is only overwritten for that user:
    $this->drupalGet('/user/2');
    $session->statusCodeEquals(200);

    $session->elementTextNotContains('css', 'h1', 'Overwritten name');

    // Check that the metatag is not present in the HTML head:
    $session->elementNotExists('css', 'meta[name="page_heading"]');
  }

  /**
   * Test page heading metatag using the "Taxonomy Term Metatag" default.
   */
  public function testPageHeadingOnTaxonomyTermThroughTaxonomyTermMetatagDefault() {
    $session = $this->assertSession();
    $page = $this->getSession()->getPage();

    $vocabularyBundle = $this->createVocabulary();

    // Add page heading metatag content:
    $this->drupalGet('/admin/config/search/metatag/taxonomy_term');
    $page->fillField('page_heading', 'Overwritten name');
    $page->pressButton('edit-submit');
    $session->statusCodeEquals(200);
    $session->pageTextContains('Saved the Taxonomy Term Metatag defaults.');

    // Create a new term:
    $term = $this->createTerm($vocabularyBundle, ['name' => 'My new Term']);

    // Go to the user and check the page heading:
    $this->drupalGet($term->toUrl()->toString());
    $session->statusCodeEquals(200);

    $session->elementTextEquals('css', 'h1', 'Overwritten name');

    // Check that the metatag is not present in the HTML head:
    $session->elementNotExists('css', 'meta[name="page_heading"]');
  }

  /**
   * Test page heading metatag using the metatag field.
   *
   * @todo We get a strange error, when submitting a new term in line 266:
   *
   * The "delete" plugin does not exist. Valid plugin IDs for
   * Drupal\metatag\MetatagTagPluginManager are: author, description, google,
   * set_cookie, expires, geo_region, abstract, shortlink, geo_position, next,
   * rating, robots, geo_placename, canonical_url, revisit_after, image_src,
   * original_source, refresh, rights, keywords, generator, cache_control, prev,
   * referrer, icbm, title, pragma, page_heading
   */
  public function todoTestPageHeadingOnTaxonomyTermThroughMetatagField() {
    $session = $this->assertSession();
    $page = $this->getSession()->getPage();

    $vocabularyBundle = $this->createVocabulary(['name' => 'Tags', 'vid' => 'tags']);
    $this->drupalGet('/admin/structure/taxonomy/manage/tags');
    // "fieldUIAddNewField" doesn't work for taxonomy terms, so we have to
    // create the field manually:
    FieldStorageConfig::create([
      'field_name' => 'field_metatags',
      'entity_type' => 'taxonomy_term',
      'type' => 'metatag',
      'cardinality' => FieldStorageDefinitionInterface::CARDINALITY_UNLIMITED,
    ])->save();
    // Create field instance:
    FieldConfig::create([
      'label' => 'Metatags',
      'field_name' => 'field_metatags',
      'entity_type' => 'taxonomy_term',
      'bundle' => 'tags',
      'settings' => [],
    ])->save();
    /** @var \Drupal\Core\Entity\EntityDisplayRepositoryInterface $display_repository */
    $display_repository = \Drupal::service('entity_display.repository');

    $display_repository->getFormDisplay('taxonomy_term', 'tags')
      ->setComponent('field_metatags', [
        'type' => 'metatag_firehose',
        'settings' => [],
      ])
      ->save();

    // Create a term, with a page heading metatag through the ui
    // (because programmatically it doesn't properly work):
    $this->drupalGet('/admin/structure/taxonomy/manage/tags/add');
    $page->fillField('name[0][value]', 'My tag');
    $page->fillField('field_metatags[0][page_heading][page_heading]', 'Overwritten name');
    $page->pressButton('edit-submit');

    $this->assertSession()->statusCodeEquals(200);

    // Go to the term and check the page heading:
    $this->drupalGet('/taxonomy/term/1');
    $session->statusCodeEquals(200);

    $session->elementTextEquals('css', 'h1', 'Overwritten name');

    // Check that the metatag is not present in the HTML head:
    $session->elementNotExists('css', 'meta[name="page_heading"]');

    $this->createTerm($vocabularyBundle, ['name' => 'My new Term']);

    // Check, that the page heading is only overwritten for that user:
    $this->drupalGet('/taxonomy/term/2');
    $session->statusCodeEquals(200);

    $session->elementTextNotContains('css', 'h1', 'Overwritten name');

    // Check that the metatag is not present in the HTML head:
    $session->elementNotExists('css', 'meta[name="page_heading"]');
  }

  /**
   * Test page heading metatag using the "Front Page Metatag" default.
   */
  public function testFrontPageHeadingOverwritten() {
    $session = $this->assertSession();
    $page = $this->getSession()->getPage();

    // Add page heading metatag content:
    $this->drupalGet('/admin/config/search/metatag/front');
    $page->fillField('page_heading', 'Overwritten front page title');
    $page->pressButton('edit-submit');
    $session->statusCodeEquals(200);
    $session->pageTextContains('Saved the Front page Metatag defaults.');

    // Go to the front page and check the page heading:
    $this->drupalGet('<front>');
    $session->elementTextEquals('css', 'h1', 'Overwritten front page title');

    // Check that the metatag is not present in the HTML head:
    $session->elementNotExists('css', 'meta[name="page_heading"]');
  }

  /**
   * Test to see if providing no value will not overwrite the page heading.
   */
  public function testFrontPageHeadingNotOverwritten() {
    $session = $this->assertSession();
    $page = $this->getSession()->getPage();

    // Add page heading metatag content:
    $this->drupalGet('/admin/config/search/metatag/front');
    $page->fillField('page_heading', '');
    $page->pressButton('edit-submit');
    $session->statusCodeEquals(200);
    $session->pageTextContains('Saved the Front page Metatag defaults.');

    // Go to the front page and check the page heading:
    $this->drupalGet('<front>');
    // Check if the test page module front page title is present:
    $session->elementTextEquals('css', 'h1', 'Test page');

    // Check that the metatag is not present in the HTML head:
    $session->elementNotExists('css', 'meta[name="page_heading"]');
  }

}
