<?php

namespace Drupal\metatag_simple_widget\Plugin\Field\FieldWidget;

use Drupal\Component\Serialization\Json;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\WidgetBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\metatag\MetatagTagPluginManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Simplified widget for metatag fields.
 *
 * This widget provides a streamlined interface for managing basic meta tags
 * (title and description) without the complexity of the full metatag form.
 * It's designed for content editors who need simple SEO control.
 *
 * @FieldWidget(
 *   id = "metatag_simple",
 *   label = @Translation("Simplified meta tags form"),
 *   field_types = {
 *     "metatag"
 *   }
 * )
 */
class MetatagSimple extends WidgetBase implements ContainerFactoryPluginInterface {

  /**
   * Supported meta tags for this simplified widget.
   */
  private const SUPPORTED_TAGS = ['title', 'description'];

  /**
   * Maximum length for title field.
   */
  private const TITLE_MAX_LENGTH = 60;

  /**
   * Maximum length for description field.
   */
  private const DESCRIPTION_MAX_LENGTH = 135;

  /**
   * Number of rows for description textarea.
   */
  private const DESCRIPTION_ROWS = 3;

  /**
   * Instance of MetatagTagPluginManager service.
   *
   * @var \Drupal\metatag\MetatagTagPluginManager
   */
  protected $metatagPluginManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $plugin_id,
      $plugin_definition,
      $configuration['field_definition'],
      $configuration['settings'],
      $configuration['third_party_settings'],
      $container->get('plugin.manager.metatag.tag')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function __construct($plugin_id, $plugin_definition, FieldDefinitionInterface $field_definition, array $settings, array $third_party_settings, MetatagTagPluginManager $plugin_manager) {
    parent::__construct($plugin_id, $plugin_definition, $field_definition, $settings, $third_party_settings);
    $this->metatagPluginManager = $plugin_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
      'show_default_values' => TRUE,
      'help_text_general' => 'Metatags are primarily used by search engines or social media sharing to get extra information about this page.',
      'help_text_title' => 'The page title when shared on social media, or in Google Search results. Max length: @max characters.',
      'help_text_description' => 'A concise summary of the page. Might be used when shared on social media, or in Google Search results. Max length: @max characters.',
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $element['show_default_values'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show default tokens in form'),
      '#default_value' => $this->getSetting('show_default_values'),
      '#description' => $this->t('If checked, show default values such as [node:title] in the form.'),
    ];

    $element['help_text_general'] = [
      '#type' => 'textarea',
      '#title' => $this->t('General help text'),
      '#description' => $this->t('Help text displayed at the top of the metatag form.'),
      '#default_value' => $this->getSetting('help_text_general'),
      '#rows' => 2,
    ];

    $element['help_text_title'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Title field help text'),
      '#description' => $this->t('Help text displayed below the meta title field. You can use @max as a placeholder for the maximum length.'),
      '#default_value' => $this->getSetting('help_text_title'),
      '#rows' => 2,
    ];

    $element['help_text_description'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Description field help text'),
      '#description' => $this->t('Help text displayed below the meta description field. You can use @max as a placeholder for the maximum length.'),
      '#default_value' => $this->getSetting('help_text_description'),
      '#rows' => 2,
    ];

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $summary = [];

    if ($this->getSetting('show_default_values')) {
      $summary[] = $this->t('Show default values: Yes');
    }
    else {
      $summary[] = $this->t('Show default values: No');
    }

    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function formElement(FieldItemListInterface $items, $delta, array $element, array &$form, FormStateInterface $form_state) {
    $item = $items[$delta];
    $values = $this->getProcessedValues($item, $items->getEntity());

    $element['#group'] = 'advanced';
    $element['#type'] = 'details';
    $element['#open'] = FALSE;

    // Use configurable help text for general description.
    $general_help = $this->getSetting('help_text_general');
    if (!empty($general_help)) {
      $element['#description'] = $this->t($general_help);
    }

    // Use configurable help text for title field.
    $title_help = $this->getSetting('help_text_title');
    $element['title'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Meta title'),
      '#default_value' => $values['title'] ?? '',
      '#maxlength' => self::TITLE_MAX_LENGTH,
    ];
    if (!empty($title_help)) {
      $element['title']['#description'] = $this->t($title_help, ['@max' => self::TITLE_MAX_LENGTH]);
    }

    // Use configurable help text for description field.
    $description_help = $this->getSetting('help_text_description');
    $element['description'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Meta description'),
      '#default_value' => $values['description'] ?? '',
      '#rows' => self::DESCRIPTION_ROWS,
      '#maxlength' => self::DESCRIPTION_MAX_LENGTH,
    ];
    if (!empty($description_help)) {
      $element['description']['#description'] = $this->t($description_help, ['@max' => self::DESCRIPTION_MAX_LENGTH]);
    }

    return $element;
  }

  /**
   * Get decoded values and populate defaults in a single operation.
   *
   * @param mixed $item
   *   The field item.
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity.
   *
   * @return array
   *   The processed values array with defaults populated if applicable.
   */
  private function getProcessedValues($item, $entity): array {
    // Decode the stored values.
    $values = [];
    if (!empty($item->value)) {
      $values = metatag_data_decode($item->value);
      if (!is_array($values)) {
        $values = [];
      }
    }

    // Populate default values if the setting is enabled.
    if ($this->getSetting('show_default_values')) {
      $default_tags = metatag_get_default_tags($entity);
      if (!empty($default_tags)) {
        foreach ($default_tags as $tag_id => $tag_value) {
          if (!isset($values[$tag_id]) && !empty($tag_value)) {
            $values[$tag_id] = $tag_value;
          }
        }
      }
    }

    return $values;
  }

  /**
   * {@inheritdoc}
   */
  public function massageFormValues(array $values, array $form, FormStateInterface $form_state) {
    foreach ($values as &$value) {
      $flattened_value = [];

      foreach (self::SUPPORTED_TAGS as $tag_id) {
        if (!empty($value[$tag_id])) {
          $tag = $this->metatagPluginManager->createInstance($tag_id);
          $tag->setValue($value[$tag_id]);
          $tag_value = $tag->value();
          if (!empty($tag_value)) {
            $flattened_value[$tag_id] = $tag_value;
          }
        }
      }

      $value = Json::encode($flattened_value);
    }

    return $values;
  }

}
