<?php

declare(strict_types=1);

namespace Drupal\microsoft_graph_mailer;

use Psr\Http\Client\ClientInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Component\Utility\EmailValidatorInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;

/**
 * Client for interacting with the Microsoft Graph API.
 */
final class GraphClient implements GraphClientInterface {

  use StringTranslationTrait;

  /**
   * The HTTP client for making requests to the Microsoft Graph API.
   *
   * @var \Psr\Http\Client\ClientInterface
   */
  protected $httpClient;

  /**
   * The email validator.
   *
   * @var \Drupal\Component\Utility\EmailValidatorInterface
   */
  protected $emailValidator;

  /**
   * Configuration factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * The logger channel.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * The current access token data.
   *
   * @var array
   */
  protected $accessToken = [];

  /**
   * Constructs a GraphClient object.
   */
  public function __construct(
    ClientInterface $http_client,
    EmailValidatorInterface $email_validator,
    ConfigFactoryInterface $config_factory,
    ModuleHandlerInterface $module_handler,
    LoggerChannelFactoryInterface $logger,
  ) {
    $this->httpClient = $http_client;
    $this->emailValidator = $email_validator;
    $this->configFactory = $config_factory;
    $this->moduleHandler = $module_handler;
    $this->logger = $logger->get('microsoft_graph_mailer');
  }

  /**
   * {@inheritdoc}
   */
  public function getAccessToken(?string $tenant_id = NULL, ?string $client_id = NULL, ?string $client_secret = NULL): array {
    $data = [];
    // Status of mailer integration.
    $module_settings = $this->configFactory->get('microsoft_graph_mailer.settings');
    if (!$module_settings->get('enabled')) {
      return $data;
    }
    if (
      isset($this->accessToken['token_type']) &&
      isset($this->accessToken['access_token']) &&
      isset($this->accessToken['expires_in']) &&
      isset($this->accessToken['requested']) &&
      is_numeric($this->accessToken['expires_in']) &&
      is_numeric($this->accessToken['requested'])
    ) {
      if (time() < ($this->accessToken['requested'] + $this->accessToken['expires_in'])) {
        return $this->accessToken;
      }
    }
    $tenant_id = $tenant_id ?? $module_settings->get('tenant_id');
    $client_id = $client_id ?? $module_settings->get('client_id');
    $client_secret = $client_secret ?? $module_settings->get('client_secret');
    try {
      $response = $this->httpClient->post('https://login.microsoftonline.com/' . $tenant_id . '/oauth2/v2.0/token', [
        'form_params' => [
          'grant_type' => 'client_credentials',
          'client_id' => $client_id,
          'client_secret' => $client_secret,
          'scope' => 'https://graph.microsoft.com/.default',
        ],
      ]);
      $data = $response->getStatusCode() === 200 ? json_decode((string) $response->getBody(), TRUE) : [];
      if (isset($data['token_type']) && isset($data['expires_in']) && isset($data['access_token'])) {
        $this->accessToken = $data + ['requested' => time()];
      }
    }
    catch (\Exception $exception) {
      $this->logger->error($exception->getMessage() ?: $this->t('Unknown error trying to get Microsoft Graph API access token.'));
    }

    return $this->accessToken;
  }

  /**
   * Get the plain access token for the Microsoft Graph API.
   *
   * @param string $tenant_id
   *   Tenant ID.
   * @param string $client_id
   *   App Client ID.
   * @param string $client_secret
   *   App Client secret.
   */
  public function getPlainAccessToken(?string $tenant_id = NULL, ?string $client_id = NULL, ?string $client_secret = NULL): string {
    $data = $this->getAccessToken($tenant_id, $client_id, $client_secret);
    if (!isset($data['access_token'])) {
      $this->logger->error('Access token not found in response: @response', ['@response' => json_encode($data)]);
    }

    return $data['access_token'] ?? '';
  }

  /**
   * {@inheritdoc}
   */
  public function sendMail(array $message): bool {
    $result = FALSE;
    $module_settings = $this->configFactory->get('microsoft_graph_mailer.settings');
    if (!$module_settings->get('enabled')) {
      return $result;
    }
    $site_settings = $this->configFactory->get('system.site');
    // If enabled debug mode, all mail should go to the specified mail address.
    if ($module_settings->get('debug_mode')) {
      $message['to'] = $module_settings->get('debug_email') ?: ($module_settings->get('default_email') ?: $site_settings->get('mail'));
    }
    // We allow to override the API settings by passing them as message params.
    $tenant_id = $client_id = $client_secret = NULL;
    // Initializing needed variables.
    $_from = $message['from'] ?? '';
    $_subject = '';
    $_body = $_to_recipients = [];
    $_content_type = 'Text';
    if (isset($message['params'])) {
      // If passed third party API credentials, use them instead default ones.
      if (isset($message['params']['api'])) {
        $tenant_id = $message['params']['api']['tenant_id'] ?? NULL;
        $client_id = $message['params']['api']['client_id'] ?? NULL;
        $client_secret = $message['params']['api']['client_secret'] ?? NULL;
      }
      $_from = $message['params']['from'] ?? $_from;
      $_subject = $message['params']['subject'] ?? '';
      $_body = $message['params']['body'] ?? '';
      $_content_type = $message['params']['content_type'] ?? $_content_type;
    }
    if (isset($message['to']) && !empty($message['to']) && is_string($message['to'])) {
      foreach (explode(',', preg_replace('/\s+/', '', $message['to'])) as $to) {
        if ($this->emailValidator->isValid($to)) {
          array_push($_to_recipients, [
            'emailAddress' => [
              'address' => $to,
            ],
          ]);
        }
      }
    }
    if (empty($_to_recipients)) {
      return $result;
    }
    $token = $this->getPlainAccessToken($tenant_id, $client_id, $client_secret);
    $payload = [
      'message' => [
        'subject' => $_subject,
        'body' => [
          'contentType' => $_content_type,
          'content' => is_array($_body) ? implode("\n\n", $_body) : $_body,
        ],
        'toRecipients' => $_to_recipients,
      ],
      'saveToSentItems' => TRUE,
    ];
    if ($_from) {
      $payload['message'] += [
        'from' => [
          'emailAddress' => [
            'address' => $_from,
          ],
        ],
      ];
    }
    $_attachments = $this->provideMessageAttachments($message);
    if (!empty($_attachments)) {
      $payload['message'] += [
        'attachments' => $_attachments,
      ];
    }
    // External modules may alter the email parameters.
    $this->moduleHandler->invokeAll('microsoft_graph_mailer_email_data_alter', [&$payload, $message]);
    try {
      $response = $this->httpClient->post('https://graph.microsoft.com/v1.0/users/' . $_from . '/sendMail', [
        'headers' => [
          'Authorization' => "Bearer $token",
          'Content-Type'  => 'application/json',
        ],
        'json' => $payload,
      ]);
      $result = $response->getStatusCode() === 202;
    }
    catch (\Exception $exception) {
      $this->logger->error($exception->getMessage() ?: $this->t('Unknown error trying to send email.'));
    }

    return $result;
  }

  /**
   * {@inheritdoc}
   */
  public function pullMail(string $address, array $options): array {
    $data = [];
    // Validate the email address.
    if (!$this->emailValidator->isValid($address)) {
      $this->logger->error('Invalid email address provided to fetch email from: @address', ['@address' => $address]);

      return $data;
    }
    $_query = [
      '$filter' => 'isRead eq false',
      '$top'    => 10,
    ];
    if (isset($options['query']) && is_array($options['query'])) {
      $_query = array_merge($_query, $options['query']);
    }
    // Format should be always in JSON.
    $_query['$format'] = 'application/json';
    // We allow to override the API settings by passing them as options.
    $tenant_id = $client_id = $client_secret = NULL;
    if (isset($options['api'])) {
      $tenant_id = $options['api']['tenant_id'] ?? NULL;
      $client_id = $options['api']['client_id'] ?? NULL;
      $client_secret = $options['api']['client_secret'] ?? NULL;
    }
    $token = $this->getPlainAccessToken($tenant_id, $client_id, $client_secret);
    if ($token) {
      try {
        $response = $this->httpClient->get('https://graph.microsoft.com/v1.0/users/' . $address . '/mailFolders/inbox/messages', [
          'headers' => [
            'Authorization' => 'Bearer ' . $token,
            'Accept' => 'application/json',
          ],
          'query' => $_query,
        ]);
        $data = $response->getStatusCode() === 200 ? json_decode($response->getBody()->getContents(), TRUE) : [];
      }
      catch (\Exception $exception) {
        $this->logger->error($exception->getMessage() ?: $this->t('Unknown error trying to pull email from @mail.', ['@mail' => $address]));
      }
    }

    return $data;
  }

  /**
   * {@inheritdoc}
   */
  public function performMessageOperations(string $mail, string $message_id, array $options) : array {
    $data = [];
    // Validate the email address.
    if (!$this->emailValidator->isValid($mail)) {
      $this->logger->error('Invalid email address provided to perform message operation: @address', ['@address' => $mail]);

      return $data;
    }
    $_operations = [];
    if (isset($options['operations']) && is_array($options['operations'])) {
      $_operations = array_merge($_operations, $options['operations']);
    }
    if (empty($_operations)) {
      return $data;
    }
    // We allow to override the API settings by passing them as options.
    $tenant_id = $client_id = $client_secret = NULL;
    if (isset($options['api'])) {
      $tenant_id = $options['api']['tenant_id'] ?? NULL;
      $client_id = $options['api']['client_id'] ?? NULL;
      $client_secret = $options['api']['client_secret'] ?? NULL;
    }
    $token = $this->getPlainAccessToken($tenant_id, $client_id, $client_secret);
    if ($token) {
      try {
        $response = $this->httpClient->patch('https://graph.microsoft.com/v1.0/users/' . $mail . '/messages/' . $message_id, [
          'headers' => [
            'Authorization' => 'Bearer ' . $token,
            'Content-Type' => 'application/json',
          ],
          'json' => $_operations,
        ]);
        $data = $response->getStatusCode() === 200 ? json_decode($response->getBody()->getContents(), TRUE) : [];
      }
      catch (\Exception $exception) {
        $this->logger->error($exception->getMessage() ?: $this->t('Unknown error trying to perform message operation as @mail.', ['@mail' => $mail]));
      }
    }

    return $data;
  }

  /**
   * {@inheritdoc}
   */
  public function provideMessageAttachments(array $message) : array {
    $attachments = [];
    if (isset($message['params']) && isset($message['params']['attachments'])) {
      foreach ($message['params']['attachments'] as $attachment) {
        if (
          is_array($attachment) &&
          isset($attachment['filepath']) &&
          isset($attachment['filename']) &&
          isset($attachment['filemime']) &&
          file_exists($attachment['filepath'])
        ) {
          $_size = filesize($attachment['filepath']);
          if ($_size < self::EMAIL_ATTACHMENT_MAX_FILESIZE) {
            array_push(
              $attachments,
              [
                '@odata.type' => '#microsoft.graph.fileAttachment',
                'name' => $attachment['filename'],
                'contentType' => $attachment['filemime'],
                'contentBytes' => base64_encode(file_get_contents($attachment['filepath'])),
              ]
            );
          }
          else {
            // Can't upload file. Exceeds 3MB.
            $this->logger->error('Unable to attach <em>@file</em> file: Exceeds limit of @limit.', [
              '@file' => $attachment['filepath'],
              '@limit' => '3MB',
            ]);
          }
        }
      }
    }

    return $attachments;
  }

  /**
   * {@inheritdoc}
   */
  public function pullMessageAttachments(string $mail, string $message_id, array $options = []) : array {
    $data = [];
    // Validate the email address.
    if (!$this->emailValidator->isValid($mail)) {
      $this->logger->error('Invalid email address provided to pull message attachments: @address', ['@address' => $mail]);

      return $data;
    }
    // We allow to override the API settings by passing them as options.
    $tenant_id = $client_id = $client_secret = NULL;
    if (isset($options['api'])) {
      $tenant_id = $options['api']['tenant_id'] ?? NULL;
      $client_id = $options['api']['client_id'] ?? NULL;
      $client_secret = $options['api']['client_secret'] ?? NULL;
    }
    $token = $this->getPlainAccessToken($tenant_id, $client_id, $client_secret);
    if ($token) {
      try {
        $response = $this->httpClient->get('https://graph.microsoft.com/v1.0/users/' . $mail . '/messages/' . $message_id . '/attachments', [
          'headers' => [
            'Authorization' => 'Bearer ' . $token,
            'Accept' => 'application/json',
          ],
        ]);
        $data = $response->getStatusCode() === 200 ? json_decode($response->getBody()->getContents(), TRUE) : [];
      }
      catch (\Exception $exception) {
        $this->logger->error($exception->getMessage() ?: $this->t('Unknown error trying to pull message attachments as @mail.', ['@mail' => $mail]));
      }
    }

    return $data;
  }

}
