<?php

declare(strict_types=1);

namespace Drupal\microsoft_graph_mailer;

/**
 * Interface for the Microsoft Graph API client.
 */
interface GraphClientInterface {

  /**
   * The max filesize for email attachments (bytes).
   *
   * @string
   */
  const EMAIL_ATTACHMENT_MAX_FILESIZE = 3000000;

  /**
   * Get the access token for the Microsoft Graph API.
   *
   * @param string $tenant_id
   *   Tenant ID.
   * @param string $client_id
   *   App Client ID.
   * @param string $client_secret
   *   App Client secret.
   */
  public function getAccessToken(string $tenant_id, string $client_id, string $client_secret): array;

  /**
   * Send an email message using the Microsoft Graph API.
   */
  public function sendMail(array $message): bool;

  /**
   * Pull emails from the Microsoft Graph API.
   */
  public function pullMail(string $address, array $options): array;

  /**
   * Perform a specified operation into the given account and message.
   *
   * @param string $mail
   *   The account containing this message.
   * @param string $message_id
   *   The message ID to perform this operation on.
   * @param array $options
   *   Our set of options for the operation, API and others, as needed.
   */
  public function performMessageOperations(string $mail, string $message_id, array $options) : array;

  /**
   * Get attachments from a specified mailbox and message ID.
   *
   * @param string $mail
   *   The account containing this message.
   * @param string $message_id
   *   The message ID to look for attachments on.
   * @param array $options
   *   Our set of options for the operation, API and others, as needed.
   */
  public function pullMessageAttachments(string $mail, string $message_id, array $options = []) : array;

  /**
   * Provide attachments data according to data from $message array.
   *
   * @param array $message
   *   The mail message.
   */
  public function provideMessageAttachments(array $message) : array;

}
