<?php

declare(strict_types=1);

namespace Drupal\microsoft_graph_mailer\Form;

use Drupal\Core\Url;
use Drupal\Core\Link;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configure Mailer settings for this site.
 */
final class MicrosoftGraphMailerSettingsForm extends ConfigFormBase {

  /**
   * The Mailer Graph client.
   *
   * @var \Drupal\mailer\GraphClientInterface
   */
  protected $graphClient;

  /**
   * Constructs a new MailerSettingsForm.
   *
   * @param \Drupal\mailer\GraphClientInterface $graphClient
   *   The Graph client.
   */
  public function __construct($graphClient) {
    $this->graphClient = $graphClient;
  }

  /**
   * {@inheritdoc}
   */
  public static function create($container) {
    return new self(
      $container->get('microsoft_graph_mailer.graph_client')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'microsoft_graph_mailer_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['microsoft_graph_mailer.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    // Load the configuration for the mailer settings.
    $module_settings = $this->config('microsoft_graph_mailer.settings');
    $site_settings = $this->config('system.site');
    $form['enabled'] = [
      '#type' => 'select',
      '#title' => $this->t('Enable Mailer'),
      '#options' => [
        '1' => $this->t('Enabled'),
        '0' => $this->t('Disabled'),
      ],
      '#default_value' => $module_settings->get('enabled') ? '1' : '0',
      '#description' => $this->t('Enable or disable the mailer functionality.<br />It should be enabled to send emails using the Microsoft Graph API or disabled during migration.'),
    ];
    $form['debug_mode'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Debug mode'),
      '#default_value' => $module_settings->get('debug_mode'),
    ];
    $form['debug_email'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Debug mode email'),
      '#default_value' => $module_settings->get('debug_email'),
      '#states' => [
        'visible' => [
          ':input[name="debug_mode"]' => ['checked' => TRUE],
        ],
      ],
      '#description' => $this->t('Enter an email for debug purposes. If no value provided, the default site email will be used.'),
    ];
    $form['client_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Client ID'),
      '#default_value' => $module_settings->get('client_id'),
      '#description' => $this->t('<b>Application (client) ID:</b> the Client ID of the application registered in Azure AD.'),
      '#required' => TRUE,
    ];
    $form['client_secret'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Client Secret'),
      '#default_value' => $module_settings->get('client_secret'),
      '#description' => $this->t('<b>Client secret value:</b> the Client Secret of the application registered in Azure AD. This is a sensitive value and should be kept secret.'),
      '#required' => TRUE,
    ];
    $form['tenant_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Tenant ID'),
      '#default_value' => $module_settings->get('tenant_id'),
      '#description' => $this->t('<b>Directory (tenant) ID:</b> the Tenant ID of the Azure AD where the application is registered. This is used to identify the Azure AD instance.'),
      '#required' => TRUE,
    ];
    $form['default_content_type'] = [
      '#type' => 'select',
      '#title' => $this->t('Default content type'),
      '#options' => [
        'Text' => $this->t('Plain text'),
        'HTML' => $this->t('HTML'),
      ],
      '#default_value' => $module_settings->get('default_content_type') ?? 'Text',
    ];
    $form['default_email'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Default email'),
      '#default_value' => $module_settings->get('default_email'),
      '#description' => $this->t("The default email address for most operations. If no value is set, then the default site's email address (<strong>@email</strong>) from @link will be used.", [
        '@email' => $site_settings->get('mail') ?: '',
        '@link' => Link::fromTextAndUrl(
          $this->t('site information'),
          Url::fromRoute('system.site_information_settings')
        )->toString(),
      ]),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    $config = $this->config('microsoft_graph_mailer.settings');
    $status = $config->get('enabled');
    // Validate credentials if they are set.
    if ($status &&
      !empty($form_state->getValue('client_id')) &&
      !empty($form_state->getValue('client_secret')) &&
      !empty($form_state->getValue('tenant_id'))) {
      try {
        $token = $this->graphClient->getPlainAccessToken(
          $form_state->getValue('tenant_id'),
          $form_state->getValue('client_id'),
          $form_state->getValue('client_secret')
        );
        if (empty($token)) {
          $form_state->setErrorByName('client_id', $this->t('Invalid credentials: Unable to obtain access token.'));
        }
        else {
          // Let the user know that the credentials were valid.
          $this->messenger()->addStatus($this->t('Successfully connected to the mail server.'));
        }
      }
      catch (\Exception $e) {
        $form_state->setErrorByName('client_id', $this->t('Invalid credentials: @message', ['@message' => $e->getMessage()]));
      }
    }
    else {
      $this->messenger()->addStatus($this->t('Skipping connectivity check for the mail server.'));
    }

    parent::validateForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $this->config('microsoft_graph_mailer.settings')
      ->set('enabled', $form_state->getValue('enabled'))
      ->set('debug_mode', $form_state->getValue('debug_mode'))
      ->set('debug_email', $form_state->getValue('debug_email'))
      ->set('client_id', $form_state->getValue('client_id'))
      ->set('client_secret', $form_state->getValue('client_secret'))
      ->set('tenant_id', $form_state->getValue('tenant_id'))
      ->set('default_content_type', $form_state->getValue('default_content_type'))
      ->set('default_email', $form_state->getValue('default_email'))
      ->save();

    parent::submitForm($form, $form_state);
  }

}
