<?php

declare(strict_types=1);

namespace Drupal\microsoft_graph_mailer\Plugin\Mail;

use Drupal\Core\Mail\MailInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\microsoft_graph_mailer\GraphClientInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Defines a mail plugin that uses Microsoft Graph Mail.
 *
 * @Mail(
 *   id = "microsoft_graph_mail",
 *   label = @Translation("Microsoft Graph Mail")
 * )
 */
final class MicrosoftGraphMail implements MailInterface, ContainerFactoryPluginInterface {

  use StringTranslationTrait;

  /**
   * The Graph client for interacting with the Microsoft Graph API.
   *
   * @var \Drupal\microsoft_graph_mailer\GraphClientInterface
   */
  protected $graphClient;

  /**
   * {@inheritDoc}
   */
  public function __construct(GraphClientInterface $client) {
    $this->graphClient = $client;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $container->get('microsoft_graph_mailer.graph_client'),
    );
  }

  /**
   * Formats the message for sending.
   *
   * @param array $message
   *   An associative array containing the email details.
   *
   * @return array
   *   The formatted message.
   */
  public function format(array $message) {
    return $message;
  }

  /**
   * Sends an email using Microsoft Graph.
   *
   * @param array $message
   *   An associative array containing the email details.
   *
   * @return mixed
   *   The response from the Microsoft Graph API.
   */
  public function mail(array $message) {
    return $this->graphClient->sendMail($message);
  }

}
