<?php

namespace Drupal\mida_ab_testing\Form;

use Drupal\Core\Condition\ConditionManager;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Form\SubformState;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Plugin\Context\ContextRepositoryInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure Mida A/B Testing settings.
 */
final class MidaSettingsForm extends ConfigFormBase {

  /**
   * The condition plugin manager.
   *
   * @var \Drupal\Core\Condition\ConditionManager
   */
  protected $conditionManager;

  /**
   * The language manager service.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected $languageManager;

  /**
   * The context repository service.
   *
   * @var \Drupal\Core\Plugin\Context\ContextRepositoryInterface
   */
  protected $contextRepository;

  /**
   * Constructs a MidaSettingsForm object.
   *
   * @param \Drupal\Core\Condition\ConditionManager $condition_manager
   *   The condition plugin manager.
   * @param \Drupal\Core\Language\LanguageManagerInterface $language_manager
   *   The language manager service.
   * @param \Drupal\Core\Plugin\Context\ContextRepositoryInterface $context_repository
   *   The context repository service.
   */
  public function __construct(
    ConditionManager $condition_manager,
    LanguageManagerInterface $language_manager,
    ContextRepositoryInterface $context_repository,
  ) {
    $this->conditionManager = $condition_manager;
    $this->languageManager = $language_manager;
    $this->contextRepository = $context_repository;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('plugin.manager.condition'),
      $container->get('language_manager'),
      $container->get('context.repository')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'mida_ab_testing_settings';
  }

  /**
   * {@inheritdoc}
   *
   * @return array<int, string>
   *   An array of configuration object names.
   */
  protected function getEditableConfigNames(): array {
    return ['mida_ab_testing.settings'];
  }

  /**
   * {@inheritdoc}
   *
   * @param array<string, mixed> $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array<string, mixed>
   *   The form array.
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('mida_ab_testing.settings');

    $form['enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable Mida A/B Testing'),
      '#description' => $this->t('Check to enable the Mida A/B testing scripts on the site.'),
      '#default_value' => $config->get('enabled'),
    ];

    $form['mida_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Mida API Key'),
      '#description' => $this->t('Enter your Mida API key (e.g., 3Ejq4KNndxpo0dJyArpv7b).'),
      '#default_value' => $config->get('mida_key'),
      '#maxlength' => 64,
      '#states' => [
        'required' => [
          ':input[name="enabled"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['timeout'] = [
      '#type' => 'number',
      '#title' => $this->t('Anti-flicker Timeout (ms)'),
      '#description' => $this->t('Timeout value in milliseconds to remove the anti-flicker overlay. Default is 3000ms.'),
      '#default_value' => $config->get('timeout') ?: 3000,
      '#min' => 500,
      '#max' => 10000,
      '#step' => 100,
    ];

    // Build the visibility conditions UI.
    $form['visibility'] = $this->buildVisibilityInterface($form_state);

    return parent::buildForm($form, $form_state);
  }

  /**
   * Builds the visibility interface for conditions.
   *
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   *
   * @return array<string, mixed>
   *   The form array with visibility conditions.
   */
  protected function buildVisibilityInterface(FormStateInterface $form_state): array {
    $form = [
      '#type' => 'details',
      '#title' => $this->t('Visibility'),
      '#open' => TRUE,
      '#tree' => TRUE,
    ];

    $form['visibility_tabs'] = [
      '#type' => 'vertical_tabs',
      '#title' => $this->t('Visibility'),
      '#parents' => ['visibility_tabs'],
    ];

    $visibility = $this->config('mida_ab_testing.settings')->get('visibility') ?: [];

    // Get available condition plugins, filtered for block_ui context.
    $definitions = $this->conditionManager->getFilteredDefinitions(
      'block_ui',
      $form_state->getTemporaryValue('gathered_contexts') ?: $this->contextRepository->getAvailableContexts()
    );

    foreach ($definitions as $condition_id => $definition) {
      // Skip the current_theme condition as it's not relevant here.
      if ($condition_id == 'current_theme') {
        continue;
      }
      // Don't display the language condition until we have multiple languages.
      if ($condition_id == 'language' && !$this->languageManager->isMultilingual()) {
        continue;
      }

      /** @var \Drupal\Core\Condition\ConditionInterface $condition */
      $condition = $this->conditionManager->createInstance($condition_id, $visibility[$condition_id] ?? []);
      $form_state->set(['conditions', $condition_id], $condition);

      $condition_form = $condition->buildConfigurationForm([], $form_state);
      $condition_form['#type'] = 'details';
      /** @var array<string, mixed> $definition */
      $condition_form['#title'] = $definition['label'];
      $condition_form['#group'] = 'visibility_tabs';
      $form[$condition_id] = $condition_form;
    }

    // Customize the request_path condition UI.
    if (isset($form['request_path']) && is_array($form['request_path'])) {
      $form['request_path']['#title'] = $this->t('Pages');
      $form['request_path']['negate']['#type'] = 'radios';
      $form['request_path']['negate']['#default_value'] = (int) ($form['request_path']['negate']['#default_value'] ?? 0);
      $form['request_path']['negate']['#title_display'] = 'invisible';
      $form['request_path']['negate']['#options'] = [
        $this->t('Show for the listed pages'),
        $this->t('Hide for the listed pages'),
      ];
    }

    // Customize the user_role condition UI.
    if (isset($form['user_role']) && is_array($form['user_role'])) {
      $form['user_role']['#title'] = $this->t('Roles');
      unset($form['user_role']['roles']['#description']);
    }

    return $form;
  }

  /**
   * {@inheritdoc}
   *
   * @phpstan-param array<string, mixed> $form
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    parent::validateForm($form, $form_state);

    $enabled = $form_state->getValue('enabled');
    $mida_key = trim($form_state->getValue('mida_key'));

    if ($enabled && empty($mida_key)) {
      $form_state->setErrorByName('mida_key', $this->t('The Mida API Key is required when the module is enabled.'));
    }

    // Validate the key format (alphanumeric only).
    if (!empty($mida_key) && !preg_match('/^[a-zA-Z0-9]+$/', $mida_key)) {
      $form_state->setErrorByName('mida_key', $this->t('The Mida API Key should only contain alphanumeric characters.'));
    }

    // Validate visibility condition settings.
    $visibility = $form_state->getValue('visibility', []);
    foreach ($visibility as $condition_id => $values) {
      if ($condition_id === 'visibility_tabs') {
        continue;
      }
      $condition = $form_state->get(['conditions', $condition_id]);
      if ($condition) {
        /** @var array<string, mixed> $subform */
        $subform = $form['visibility'][$condition_id];
        $condition->validateConfigurationForm($subform, SubformState::createForSubform($subform, $form, $form_state));
      }
    }
  }

  /**
   * {@inheritdoc}
   *
   * @phpstan-param array<string, mixed> $form
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    // Process visibility conditions.
    $visibility = [];
    $visibility_values = $form_state->getValue('visibility', []);

    foreach ($visibility_values as $condition_id => $values) {
      if ($condition_id === 'visibility_tabs') {
        continue;
      }
      $condition = $form_state->get(['conditions', $condition_id]);
      if ($condition) {
        /** @var array<string, mixed> $subform */
        $subform = $form['visibility'][$condition_id];
        $condition->submitConfigurationForm($subform, SubformState::createForSubform($subform, $form, $form_state));
        $visibility[$condition_id] = $condition->getConfiguration();
      }
    }

    $this->config('mida_ab_testing.settings')
      ->set('enabled', $form_state->getValue('enabled'))
      ->set('mida_key', trim($form_state->getValue('mida_key')))
      ->set('timeout', $form_state->getValue('timeout'))
      ->set('visibility', $visibility)
      ->save();

    parent::submitForm($form, $form_state);
  }

}
