<?php

namespace Drupal\mida_ab_testing;

use Drupal\Component\Plugin\Exception\ContextException;
use Drupal\Component\Plugin\Exception\MissingValueContextException;
use Drupal\Core\Condition\ConditionManager;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Plugin\Context\ContextHandlerInterface;
use Drupal\Core\Plugin\Context\ContextRepositoryInterface;
use Drupal\Core\Plugin\ContextAwarePluginInterface;
use Drupal\Core\Render\Markup;
use Psr\Log\LoggerInterface;

/**
 * Service for attaching Mida A/B testing scripts to pages.
 */
class MidaScriptAttachment {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The condition plugin manager.
   *
   * @var \Drupal\Core\Condition\ConditionManager
   */
  protected $conditionManager;

  /**
   * The context repository service.
   *
   * @var \Drupal\Core\Plugin\Context\ContextRepositoryInterface
   */
  protected $contextRepository;

  /**
   * The context handler service.
   *
   * @var \Drupal\Core\Plugin\Context\ContextHandlerInterface
   */
  protected $contextHandler;

  /**
   * The logger service.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * Constructs a MidaScriptAttachment object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Condition\ConditionManager $condition_manager
   *   The condition plugin manager.
   * @param \Drupal\Core\Plugin\Context\ContextRepositoryInterface $context_repository
   *   The context repository service.
   * @param \Drupal\Core\Plugin\Context\ContextHandlerInterface $context_handler
   *   The context handler service.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger service.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    ConditionManager $condition_manager,
    ContextRepositoryInterface $context_repository,
    ContextHandlerInterface $context_handler,
    LoggerInterface $logger,
  ) {
    $this->configFactory = $config_factory;
    $this->conditionManager = $condition_manager;
    $this->contextRepository = $context_repository;
    $this->contextHandler = $context_handler;
    $this->logger = $logger;
  }

  /**
   * Attaches Mida scripts to page attachments.
   *
   * @param array<string, mixed> $attachments
   *   The page attachments array.
   */
  public function attach(array &$attachments): void {
    $config = $this->configFactory->get('mida_ab_testing.settings');

    // Check if the module is enabled.
    if (!$config->get('enabled')) {
      return;
    }

    // Check if we have a valid API key.
    $mida_key = $config->get('mida_key');
    if (empty($mida_key)) {
      return;
    }

    // Evaluate visibility conditions.
    if (!$this->evaluateVisibility($config->get('visibility') ?: [])) {
      return;
    }

    $timeout = $config->get('timeout') ?: 3000;

    // Anti-flickering inline script.
    $anti_flicker_script = sprintf(
      'var timeout = %d; !function(h,i,d,e){var t,n=h.createElement("style");n.id=e,n.innerHTML="body{opacity:0}",h.head.appendChild(n),t=d,i.rmfk=function(){var t=h.getElementById(e);t&&t.parentNode.removeChild(t)},setTimeout(i.rmfk,t)}(document,window,timeout,"abhide");',
      (int) $timeout
    );

    // Add the anti-flickering script first (must be inline and early).
    $attachments['#attached']['html_head'][] = [
      [
        '#type' => 'html_tag',
        '#tag' => 'script',
        '#value' => Markup::create($anti_flicker_script),
        '#attributes' => [],
        '#weight' => -1000,
      ],
      'mida_ab_testing_anti_flicker',
    ];

    // Add the Mida optimize script.
    $attachments['#attached']['html_head'][] = [
      [
        '#type' => 'html_tag',
        '#tag' => 'script',
        '#attributes' => [
          'type' => 'text/javascript',
          'async' => TRUE,
          'src' => 'https://cdn.mida.so/js/optimize.js?key=' . $mida_key,
        ],
        '#weight' => -999,
      ],
      'mida_ab_testing_optimize',
    ];

    // Add cache tags so pages are invalidated when config changes.
    $attachments['#cache']['tags'][] = 'config:mida_ab_testing.settings';
  }

  /**
   * Evaluates visibility conditions for the Mida scripts.
   *
   * @param array<string, mixed> $visibility_config
   *   The visibility configuration array.
   *
   * @return bool
   *   TRUE if all conditions pass (scripts should be shown), FALSE otherwise.
   */
  protected function evaluateVisibility(array $visibility_config): bool {
    // If no conditions are configured, show the scripts everywhere.
    if (empty($visibility_config)) {
      return TRUE;
    }

    foreach ($visibility_config as $condition_id => $condition_config) {
      // Skip if the condition config is empty or only has default values.
      if (empty($condition_config) || $this->conditionIsEmpty($condition_id, $condition_config)) {
        continue;
      }

      try {
        /** @var \Drupal\Core\Condition\ConditionInterface $condition */
        $condition = $this->conditionManager->createInstance($condition_id, $condition_config);

        // Apply context if the condition is context-aware.
        if ($condition instanceof ContextAwarePluginInterface) {
          try {
            $contexts = $this->contextRepository->getRuntimeContexts(array_values($condition->getContextMapping()));
            $this->contextHandler->applyContextMapping($condition, $contexts);
          }
          catch (MissingValueContextException $e) {
            // Context exists but has no value, condition fails.
            return FALSE;
          }
          catch (ContextException $e) {
            // Context missing, treat as condition fail.
            return FALSE;
          }
        }

        // Execute the condition and check if it passes.
        $pass = $condition->execute();

        // If condition fails (using AND logic), don't show scripts.
        if (!$pass) {
          return FALSE;
        }
      }
      catch (\Exception $e) {
        // Log the error and continue (fail safe - show scripts).
        $this->logger->warning('Error evaluating condition @id: @message', [
          '@id' => $condition_id,
          '@message' => $e->getMessage(),
        ]);
      }
    }

    return TRUE;
  }

  /**
   * Checks if a condition configuration is effectively empty.
   *
   * @param string $condition_id
   *   The condition plugin ID.
   * @param array<string, mixed> $config
   *   The condition configuration.
   *
   * @return bool
   *   TRUE if the condition is empty and should be skipped.
   */
  protected function conditionIsEmpty(string $condition_id, array $config): bool {
    switch ($condition_id) {
      case 'request_path':
        return empty($config['pages']);

      case 'user_role':
        return empty($config['roles']);

      case 'language':
        return empty($config['langcodes']);

      case 'response_status':
        return empty($config['status_codes']);

      case 'response_code':
        return empty($config['response_codes']);

      case 'webform':
        return empty($config['webforms']);

      default:
        // Handle entity_bundle:* conditions.
        if (str_starts_with($condition_id, 'entity_bundle:')) {
          return empty($config['bundles']);
        }
        // By default, skip unknown conditions with no meaningful config.
        return TRUE;
    }
  }

}
