<?php

namespace Drupal\Tests\mida\Unit;

use Drupal\Core\Condition\ConditionManager;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Plugin\Context\ContextHandlerInterface;
use Drupal\Core\Plugin\Context\ContextRepositoryInterface;
use Drupal\mida\MidaScriptAttachment;
use Drupal\Tests\UnitTestCase;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\DataProvider;
use PHPUnit\Framework\Attributes\Group;
use Psr\Log\LoggerInterface;

/**
 * Tests the MidaScriptAttachment service.
 *
 * @coversDefaultClass \Drupal\mida\MidaScriptAttachment
 * @group mida
 */
#[CoversClass(MidaScriptAttachment::class)]
#[Group('mida')]
class MidaScriptAttachmentTest extends UnitTestCase {

  /**
   * The config factory mock.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $configFactory;

  /**
   * The condition manager mock.
   *
   * @var \Drupal\Core\Condition\ConditionManager|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $conditionManager;

  /**
   * The context repository mock.
   *
   * @var \Drupal\Core\Plugin\Context\ContextRepositoryInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $contextRepository;

  /**
   * The context handler mock.
   *
   * @var \Drupal\Core\Plugin\Context\ContextHandlerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $contextHandler;

  /**
   * The logger mock.
   *
   * @var \Psr\Log\LoggerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $logger;

  /**
   * The service under test.
   *
   * @var \Drupal\mida\MidaScriptAttachment
   */
  protected $midaScriptAttachment;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->configFactory = $this->createMock(ConfigFactoryInterface::class);
    $this->conditionManager = $this->createMock(ConditionManager::class);
    $this->contextRepository = $this->createMock(ContextRepositoryInterface::class);
    $this->contextHandler = $this->createMock(ContextHandlerInterface::class);
    $this->logger = $this->createMock(LoggerInterface::class);

    $this->midaScriptAttachment = new MidaScriptAttachment(
      $this->configFactory,
      $this->conditionManager,
      $this->contextRepository,
      $this->contextHandler,
      $this->logger
    );
  }

  /**
   * Tests attach() when module is disabled.
   */
  public function testAttachWhenDisabled(): void {
    $config = $this->createMock(ImmutableConfig::class);
    $config->method('get')
      ->willReturnCallback(fn(string $key) => match ($key) {
        'enabled' => FALSE,
        default => NULL,
      });

    $this->configFactory->method('get')
      ->with('mida.settings')
      ->willReturn($config);

    $attachments = [];
    $this->midaScriptAttachment->attach($attachments);

    $this->assertArrayNotHasKey('#attached', $attachments);
  }

  /**
   * Tests attach() when API key is empty.
   */
  public function testAttachWithEmptyApiKey(): void {
    $config = $this->createMock(ImmutableConfig::class);
    $config->method('get')
      ->willReturnCallback(fn(string $key) => match ($key) {
        'enabled' => TRUE,
        'mida_key' => '',
        default => NULL,
      });

    $this->configFactory->method('get')
      ->with('mida.settings')
      ->willReturn($config);

    $attachments = [];
    $this->midaScriptAttachment->attach($attachments);

    $this->assertArrayNotHasKey('#attached', $attachments);
  }

  /**
   * Tests attach() with valid configuration.
   */
  public function testAttachWithValidConfig(): void {
    $config = $this->createMock(ImmutableConfig::class);
    $config->method('get')
      ->willReturnCallback(fn(string $key) => match ($key) {
        'enabled' => TRUE,
        'mida_key' => 'test_api_key_123',
        'visibility' => [],
        'timeout' => 3000,
        default => NULL,
      });

    $this->configFactory->method('get')
      ->with('mida.settings')
      ->willReturn($config);

    $attachments = [];
    $this->midaScriptAttachment->attach($attachments);

    // Verify attachments were added.
    $this->assertArrayHasKey('#attached', $attachments);
    $this->assertArrayHasKey('html_head', $attachments['#attached']);
    $this->assertCount(2, $attachments['#attached']['html_head']);

    // Verify anti-flicker script.
    $anti_flicker = $attachments['#attached']['html_head'][0];
    $this->assertEquals('mida_anti_flicker', $anti_flicker[1]);
    $this->assertEquals('script', $anti_flicker[0]['#tag']);
    $this->assertStringContainsString('var timeout = 3000', (string) $anti_flicker[0]['#value']);

    // Verify optimize script.
    $optimize = $attachments['#attached']['html_head'][1];
    $this->assertEquals('mida_optimize', $optimize[1]);
    $this->assertEquals('script', $optimize[0]['#tag']);
    $this->assertStringContainsString('test_api_key_123', $optimize[0]['#attributes']['src']);

    // Verify cache tags.
    $this->assertContains('config:mida.settings', $attachments['#cache']['tags']);
  }

  /**
   * Tests attach() with custom timeout.
   */
  public function testAttachWithCustomTimeout(): void {
    $config = $this->createMock(ImmutableConfig::class);
    $config->method('get')
      ->willReturnCallback(fn(string $key) => match ($key) {
        'enabled' => TRUE,
        'mida_key' => 'test_key',
        'visibility' => [],
        'timeout' => 5000,
        default => NULL,
      });

    $this->configFactory->method('get')
      ->with('mida.settings')
      ->willReturn($config);

    $attachments = [];
    $this->midaScriptAttachment->attach($attachments);

    $anti_flicker = $attachments['#attached']['html_head'][0];
    $this->assertStringContainsString('var timeout = 5000', (string) $anti_flicker[0]['#value']);
  }

  /**
   * Tests conditionIsEmpty() method via reflection.
   *
   * @param string $condition_id
   *   The condition plugin ID.
   * @param array<string, mixed> $config
   *   The condition configuration.
   * @param bool $expected
   *   The expected result.
   *
   * @dataProvider conditionIsEmptyProvider
   */
  #[DataProvider('conditionIsEmptyProvider')]
  public function testConditionIsEmpty(string $condition_id, array $config, bool $expected): void {
    $method = new \ReflectionMethod(MidaScriptAttachment::class, 'conditionIsEmpty');
    $method->setAccessible(TRUE);

    $result = $method->invoke($this->midaScriptAttachment, $condition_id, $config);
    $this->assertEquals($expected, $result);
  }

  /**
   * Data provider for testConditionIsEmpty().
   *
   * @return array<string, array{string, array<string, mixed>, bool}>
   *   Test cases.
   */
  public static function conditionIsEmptyProvider(): array {
    return [
      'request_path empty' => [
        'request_path',
        ['pages' => ''],
        TRUE,
      ],
      'request_path with pages' => [
        'request_path',
        ['pages' => '/admin/*'],
        FALSE,
      ],
      'user_role empty' => [
        'user_role',
        ['roles' => []],
        TRUE,
      ],
      'user_role with roles' => [
        'user_role',
        ['roles' => ['authenticated']],
        FALSE,
      ],
      'language empty' => [
        'language',
        ['langcodes' => []],
        TRUE,
      ],
      'language with codes' => [
        'language',
        ['langcodes' => ['en']],
        FALSE,
      ],
      'entity_bundle empty' => [
        'entity_bundle:node',
        ['bundles' => []],
        TRUE,
      ],
      'entity_bundle with bundles' => [
        'entity_bundle:node',
        ['bundles' => ['article']],
        FALSE,
      ],
      'unknown condition' => [
        'unknown_condition',
        ['some_value' => 'test'],
        TRUE,
      ],
    ];
  }

}
