<?php

declare(strict_types = 1);

namespace Drupal\migrate_default_content_export;

use Drupal\Component\Plugin\PluginBase;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Field\FieldDefinitionInterface;

/**
 * Base class for migrate_default_content_export_field_processor plugins.
 */
abstract class FieldProcessorPluginBase extends PluginBase implements FieldProcessorInterface {

  /**
   * {@inheritdoc}
   */
  public function label(): string {
    // Cast the label to a string since it is a TranslatableMarkup object.
    return (string) $this->pluginDefinition['label'];
  }

  /**
   * {@inheritdoc}
   */
  public function isApplicable(ContentEntityInterface $entity, string $field_name, FieldDefinitionInterface $field_definition): bool {
    // By default all field processors are applicable to all fields.
    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function getValue(ContentEntityInterface $entity, string $field_name, FieldDefinitionInterface $field_definition): mixed {
    $value = $entity->get($field_name)->getValue();

    return $this->process($value, $field_definition);
  }

  /**
   * Process the value of a field to retrieve the raw value.
   *
   * @param mixed $value
   *   The value to process.
   * @param \Drupal\Core\Field\FieldDefinitionInterface $field_definition
   *   The field definition of the field to process.
   *
   * @return mixed
   *   The processed value.
   */
  protected function process(mixed $value, FieldDefinitionInterface $field_definition): mixed {
    // If the value is empty, return NULL.
    if (empty($value) || !is_array($value)) {
      return NULL;
    }

    // Create a new array to store the processed values.
    $values = [];

    // Process each item in the value array.
    foreach ($value as $item) {
      if (empty($item['value'])) {
        continue;
      }

      $values[] = $item['value'];
    }

    // If the processed values array is empty, return NULL.
    if (empty($values)) {
      return NULL;
    }

    return count($values) === 1 ? reset($values) : $values;
  }

}
