<?php

declare(strict_types = 1);

namespace Drupal\migrate_default_content_export\Plugin\FieldProcessor;

use Drupal\Core\Config\Entity\ConfigEntityStorageInterface;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\migrate_default_content_export\FieldProcessorPluginBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * An entity reference field processor plugin.
 *
 * @FieldProcessor(
 *   id = "entity_reference_field_processor",
 *   label = @Translation("Entity Reference Field Processor"),
 *   description = @Translation("A field processor that replaces entity reference fields with the uuid of the target entity.")
 * )
 */
class EntityReferenceFieldProcessor extends FieldProcessorPluginBase implements ContainerFactoryPluginInterface {

  /**
   * Constructs a new FileFieldProcessor object.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, private readonly EntityTypeManagerInterface $entityTypeManager) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function isApplicable(ContentEntityInterface $entity, string $field_name, FieldDefinitionInterface $field_definition): bool {
    // Entity reference and entity reference revisions fields are applicable.
    $is_entity_reference = $field_definition->getType() === 'entity_reference';
    $is_entity_reference_revisions = $field_definition->getType() === 'entity_reference_revisions';

    // If the field is not an entity reference or entity reference revisions
    // field, return false.
    if (!($is_entity_reference || $is_entity_reference_revisions)) {
      return FALSE;
    }

    // Entity reference fields that are referencing users are not applicable.
    // This is due to not exporting the anonymous or super admin user.
    $settings = $field_definition->getSettings();
    $is_not_user = !empty($settings['target_type']) && $settings['target_type'] !== 'user';

    return $is_not_user;
  }

  /**
   * {@inheritdoc}
   */
  protected function process(mixed $value, FieldDefinitionInterface $field_definition): mixed {
    // If the value is empty, return NULL.
    if (empty($value) || !is_array($value)) {
      return NULL;
    }

    // Create a new array to store the processed values.
    $values = [];

    // Get the storage related to the target type for processing.
    $settings = $field_definition->getSettings();

    if (empty($settings['target_type'])) {
      return NULL;
    }

    $entity_storage = $this->entityTypeManager->getStorage($settings['target_type']);

    // Check if the target type is a config entity.
    $is_config_storage = $entity_storage instanceof ConfigEntityStorageInterface;

    // Process each item in the value array.
    foreach ($value as $item) {
      if (empty($item['target_id'])) {
        continue;
      }

      if ($is_config_storage) {
        $values[] = $item['target_id'];
      }
      else {
        $target_entity = $entity_storage->load($item['target_id']);

        if (empty($target_entity)) {
          continue;
        }

        $values[] = $target_entity->uuid();
      }
    }

    // If the processed values array is empty, return NULL.
    if (empty($values)) {
      return NULL;
    }

    return count($values) === 1 ? reset($values) : $values;
  }

}
