<?php

declare(strict_types = 1);

namespace Drupal\migrate_default_content_export\Plugin\FieldProcessor;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\migrate_default_content_export\FieldProcessorPluginBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * A file field processor plugin.
 *
 * @FieldProcessor(
 *   id = "file_field_processor",
 *   label = @Translation("File Field Processor"),
 *   description = @Translation("A field processor that returns the file name used in a field.")
 * )
 */
class FileFieldProcessor extends FieldProcessorPluginBase implements ContainerFactoryPluginInterface {


  /**
   * The default content directory.
   */
  private string $defaultContentDirectory;

  /**
   * The export directory.
   */
  private string $exportDirectory;

  /**
   * Constructs a new FileFieldProcessor object.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    private readonly EntityTypeManagerInterface $entityTypeManager,
    private readonly FileSystemInterface $fileSystem,
    private readonly ConfigFactoryInterface $configFactory
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);

    // Set the directories.
    $this->defaultContentDirectory = $this->configFactory->get('migrate_default_content.settings')->get('source_dir');
    $this->exportDirectory = $this->configFactory->get('migrate_default_content.settings')->get('content_export_dir');
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
      $container->get('file_system'),
      $container->get('config.factory'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function isApplicable(ContentEntityInterface $entity, string $field_name, FieldDefinitionInterface $field_definition): bool {
    // File and image fields are applicable.
    $field_types = ['file', 'image'];

    return in_array($field_definition->getType(), $field_types);
  }

  /**
   * {@inheritdoc}
   */
  protected function process(mixed $value, FieldDefinitionInterface $field_definition): mixed {
    // If the value is empty, return NULL.
    if (empty($value) || !is_array($value)) {
      return NULL;
    }

    // Create a new array to store the processed values.
    $values = [];

    // Get the storage related to the target type for processing.
    $settings = $field_definition->getSettings();

    if (empty($settings['target_type'])) {
      return NULL;
    }

    $entity_storage = $this->entityTypeManager->getStorage($settings['target_type']);

    // Process each item in the value array.
    foreach ($value as $item) {
      if (empty($item['target_id'])) {
        continue;
      }

      /** @var \Drupal\file\FileInterface $file */
      $file = $entity_storage->load($item['target_id']);

      if (!$file) {
        continue;
      }

      $filename = $file->getFilename();
      $file_uri = $file->getFileUri();
      $file_path = $this->fileSystem->realpath($file_uri);

      // If the file does not exist, skip it.
      if (!file_exists($file_path)) {
        continue;
      }

      $file_contents = file_get_contents($file_path);

      // If the file is empty, skip it.
      if (empty($file_contents)) {
        continue;
      }

      // Export the file to the default content files directory.
      $destination = $this->defaultContentDirectory . '/' . $this->exportDirectory . '/files';

      // Create the directory if it does not exist.
      if (!file_exists($destination)) {
        $this->fileSystem->prepareDirectory($destination, FileSystemInterface::CREATE_DIRECTORY);
      }

      file_put_contents($destination . '/' . $filename, $file_contents);

      $values[] = $filename;
    }

    // If the processed values array is empty, return NULL.
    if (empty($values)) {
      return NULL;
    }

    return count($values) === 1 ? reset($values) : $values;
  }

}
