<?php

declare(strict_types = 1);

namespace Drupal\migrate_default_content_export\Plugin\FieldProcessor;

use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\migrate_default_content_export\FieldProcessorPluginBase;

/**
 * An integer field processor plugin.
 *
 * @FieldProcessor(
 *   id = "integer_field_processor",
 *   label = @Translation("Integer Field Processor"),
 *   description = @Translation("A field processor that returns the integer value of the field.")
 * )
 */
class IntegerFieldProcessor extends FieldProcessorPluginBase {

  /**
   * {@inheritdoc}
   */
  public function isApplicable(ContentEntityInterface $entity, string $field_name, FieldDefinitionInterface $field_definition): bool {
    // Integer fields are applicable.
    return $field_definition->getType() === 'integer';
  }

  /**
   * {@inheritdoc}
   */
  protected function process(mixed $value, FieldDefinitionInterface $field_definition): mixed {
    // If the value is empty, return NULL.
    if (empty($value) || !is_array($value)) {
      return 0;
    }

    // Create a new array to store the processed values.
    $values = [];

    // Process each item in the value array.
    foreach ($value as $item) {
      if (empty($item['value'])) {
        $values[] = 0;
      }
      else {
        $values[] = $item['value'];
      }
    }

    // If the processed values array is empty, return NULL.
    if (empty($values)) {
      return 0;
    }

    return count($values) === 1 ? reset($values) : $values;
  }

}
