<?php

declare(strict_types = 1);

namespace Drupal\migrate_default_content_export\Plugin\FieldProcessor;

use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\migrate_default_content_export\FieldProcessorPluginBase;

/**
 * A link field processor plugin.
 *
 * @FieldProcessor(
 *   id = "link_field_processor",
 *   label = @Translation("Link Field Processor"),
 *   description = @Translation("A field processor that returns the uri and title of the link field.")
 * )
 */
final class LinkFieldProcessor extends FieldProcessorPluginBase {

  /**
   * {@inheritdoc}
   */
  public function isApplicable(ContentEntityInterface $entity, string $field_name, FieldDefinitionInterface $field_definition): bool {
    // Link fields are applicable.
    return $field_definition->getType() === 'link';
  }

  /**
   * {@inheritdoc}
   */
  protected function process(mixed $value, FieldDefinitionInterface $field_definition): mixed {
    // If the value is empty, return NULL.
    if (empty($value) || !is_array($value)) {
      return NULL;
    }

    // Create a new array to store the processed values.
    $values = [];

    // Process each item in the value array.
    foreach ($value as $item) {
      if (empty($item['uri'])) {
        $values[] = NULL;
      }
      else {
        $values[] = [
          'uri' => $item['uri'],
          'title' => $item['title'],
          'options' => $item['options'] ?? NULL,
        ];
      }
    }

    // If the processed values array is empty, return NULL.
    if (empty($values)) {
      return NULL;
    }

    return count($values) === 1 ? reset($values) : $values;
  }

}
