<?php

declare(strict_types = 1);

namespace Drupal\migrate_default_content_export\Plugin\FieldProcessor;

use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\migrate_default_content_export\FieldProcessorPluginBase;

/**
 * A text field processor plugin.
 *
 * @FieldProcessor(
 *   id = "text_field_processor",
 *   label = @Translation("Text Field Processor"),
 *   description = @Translation("A field processor that returns the format, the summary, and the value of text fields.")
 * )
 */
final class TextFieldProcessor extends FieldProcessorPluginBase {

  /**
   * {@inheritdoc}
   */
  public function isApplicable(ContentEntityInterface $entity, string $field_name, FieldDefinitionInterface $field_definition): bool {
    // Text fields are applicable.
    $allowed_field_types = ['text_with_summary', 'text_long', 'text'];

    $is_text_field = in_array($field_definition->getType(), $allowed_field_types);

    return $is_text_field;
  }

  /**
   * {@inheritdoc}
   */
  protected function process(mixed $value, FieldDefinitionInterface $field_definition): mixed {
    // If the value is empty, return NULL.
    if (empty($value) || !is_array($value)) {
      return NULL;
    }

    // Create a new array to store the processed values.
    $values = [];

    // Process each item in the value array.
    foreach ($value as $item) {
      // If the value is empty, return NULL.
      if (empty($item['value'])) {
        continue;
      }

      // Add the processed value to the values array.
      $values[] = $item;
    }

    // If the processed values array is empty, return NULL.
    if (empty($values)) {
      return NULL;
    }

    return count($values) === 1 ? reset($values) : $values;
  }

}
