<?php

namespace Drupal\migrate_default_content_export\Drush\Commands;

use Drupal\Component\Serialization\Yaml;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\migrate_default_content_export\ExportEntityFilterPluginManager;
use Drupal\migrate_default_content_export\FieldProcessorPluginManager;
use Drush\Attributes as CLI;
use Drush\Commands\DrushCommands;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * A Drush commandfile.
 *
 * In addition to this file, you need a drush.services.yml
 * in root of your module, and a composer.json file that provides the name
 * of the services file to use.
 */
final class MigrateDefaultContentExportCommands extends DrushCommands {

  /**
   * The default ignored fields.
   *
   * @var array
   */
  const DEFAULT_IGNORED_FIELDS = [
    // Bundle and type are handled by the exported file names.
    'bundle',
    'type',
    // Revision fields shouldn't be exported.
    'revision_uid',
    'revision_timestamp',
    'revision_log',
    'revision_default',
    'revision_translation_affected',
    'revision_created',
    'revision_user',
    'revision_log_message',
    'revision_id',
    'vid',
    // Created/changed fields shouldn't be exported.
    // I'm not sure about this one. Maybe make it an option?
    'created',
    'changed',
    // Language is handled by the exported file names.
    'default_langcode',
    'langcode',
    // Access and login timestamps shouldn't be exported.
    'access',
    'login',
    // Parent information (mainly for paragraphs) is handled by
    // the parent and config.
    'parent_id',
    'parent_type',
    'parent_field_name',
    // Content translation fields are handled through the file name when needed.
    'content_translation_source',
    'content_translation_outdated',
    'content_translation_uid',
    'content_translation_created',
  ];

  /**
   * The entity export filters to apply to each entity type.
   *
   * @var array
   */
  private $entityExportFilters = [];

  /**
   * The field processors to apply to each field.
   *
   * @var array
   */
  private $fieldProcessors = [];

  /**
   * The associated field processors.
   *
   * When a field processor is found to be applicable to a field type
   * it is cached here for later use.
   *
   * @var array
   */
  private $fieldProcessorCache = [];

  /**
   * The prcoessed content to export.
   *
   * @var array
   */
  private $content = [];

  /**
   * The entity types to export.
   *
   * @var array
   */
  private $entityTypes = [];

  /**
   * The bundles to export.
   *
   * @var array
   */
  private $bundles = [];

  /**
   * The IDs to export.
   *
   * @var array
   */
  private $ids = [];

  /**
   * The IDs to exclude from export.
   *
   * @var array
   */
  private $excludeIds = [];

  /**
   * The export directory.
   *
   * @var string
   */
  private $exportDir;

  /**
   * Any error messages that occur during export.
   *
   * @var string
   */
  private $errorMessage;

  /**
   * Constructs a MigrateDefaultContentExportCommands object.
   */
  public function __construct(
    private readonly EntityTypeManagerInterface $entityTypeManager,
    private readonly EntityTypeBundleInfoInterface $entityTypeBundleInfo,
    private readonly EntityFieldManagerInterface $entityFieldManager,
    private readonly Yaml $yaml,
    private readonly ConfigFactoryInterface $configFactory,
    private readonly FieldProcessorPluginManager $fieldProcessorPluginManager,
    private readonly ExportEntityFilterPluginManager $exportEntityFilterPluginManager,
  ) {
    parent::__construct();
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('entity_type.bundle.info'),
      $container->get('entity_field.manager'),
      $container->get('serialization.yaml'),
      $container->get('config.factory'),
      $container->get('plugin.manager.migrate_default_content_export.field_processor'),
      $container->get('plugin.manager.migrate_default_content_export.export_entity_filter'),
    );
  }

  /**
   * Exports content into yml.
   */
  #[CLI\Command(name: 'migrate-default-content:export-content', aliases: ['mdcec'])]
  #[CLI\Option(name: 'export', description: 'A comma-separated list of entity and bundle machine name combinations to include in the export.')]
  #[CLI\Option(name: 'exclude', description: 'A comma-separated list of entity and bundle machine name combinations to exclude from the export.')]
  #[CLI\Option(name: 'ids', description: 'A comma-separated list of entity ids to export.')]
  #[CLI\Option(name: 'exclude-ids', description: 'A comma-separated list of entity ids to exclude from the export.')]
  #[CLI\Option(name: 'allow-null', description: 'Allows null field values to be exported.')]
  #[CLI\Usage(name: 'migrate-default-content:export-content', description: 'Export all content.')]
  #[CLI\Usage(name: 'migrate-default-content:export-content --export=node', description: 'Export all nodes.')]
  #[CLI\Usage(name: 'migrate-default-content:export-content --exclude=node', description: 'Export all content except nodes.')]
  #[CLI\Usage(name: 'migrate-default-content:export-content --export=node:article', description: 'Export all articles.')]
  #[CLI\Usage(name: 'migrate-default-content:export-content --export=node --exclude="node:article"', description: 'Export all nodes except for articles.')]
  #[CLI\Usage(name: 'migrate-default-content:export-content --export=node --ids=1,4', description: 'Export nodes with the ids of 1 and 4.')]
  #[CLI\Usage(name: 'migrate-default-content:export-content --export=node --exclude-ids=1,4', description: 'Export all nodes except the ones with ids of 1 and 4.')]
  #[CLI\Usage(name: 'migrate-default-content:export-content --allow-null', description: 'Export all content without removing null fields.')]
  public function export($options = [
    'export' => 'all',
    'exclude' => 'none',
    'ids' => 'all',
    'exclude-ids' => 'none',
    'allow-null' => FALSE,
  ]) {
    $this->logger()->notice(dt('Starting export...'));

    // Set the entities/bundles to export.
    $this->logger()->notice(dt('Setting entities and bundles to export...'));
    if ($options['export'] !== 'all') {
      $exports = explode(',', $options['export']);

      foreach ($exports as $export) {
        // Split out the option into entity/bundle.
        $export = $this->processOption($export);

        // Validate the entity.
        if (!$this->validateEntity($export['entity'])) {
          $this->logger()->error($this->errorMessage);
          return;
        }

        // If a bundle is specified, validate and add it.
        // Otherwise add _all_ available bundles for the entity.
        if ($export['bundle']) {
          // Validate the entity is bundleable.
          if (!$this->validateBundleableEntity($export['entity'])) {
            $this->logger()->error($this->errorMessage);
            return;
          }

          // Validate the bundle exists.
          if (!$this->validateBundle($export['entity'], $export['bundle'])) {
            $this->logger()->error($this->errorMessage);
            return;
          }

          $this->addBundle($export['entity'], $export['bundle']);
        }
        elseif ($this->validateBundleableEntity($export['entity'])) {
          $this->addBundles($export['entity']);
        }

        // Add the entity to the list of exports.
        $this->addEntity($export['entity']);
      }
    }
    else {
      $entity_definitions = $this->entityTypeManager->getDefinitions();

      foreach ($entity_definitions as $entity_definition) {
        // Validate the entity.
        if (!$this->validateEntity($entity_definition->id())) {
          continue;
        }

        // Add the entity to the list of exports.
        $this->addEntity($entity_definition->id());

        // If the content isn't bundleable, continue.
        // Otherwise add all bundles to the list of bundles to export.
        if (!$this->validateBundleableEntity($entity_definition->id())) {
          continue;
        }

        $this->addBundles($entity_definition->id());
      }
    }

    // Filter out any excluded entities/bundles.
    if ($options['exclude'] !== 'none') {
      $excludes = explode(',', $options['exclude']);

      foreach ($excludes as $exclude) {
        // Split out the option into entity/bundle.
        $exclude = $this->processOption($exclude);

        // If no bundle is specified, remove the entity.
        // Otherwise remove the bundle.
        if (!$exclude['bundle']) {
          $this->removeEntity($exclude['entity']);
        }
        else {
          $this->removeBundle($exclude['entity'], $exclude['bundle']);
        }
      }
    }

    // Set the ids to export.
    if ($options['ids'] !== 'all') {
      $ids = explode(',', $options['ids']);

      foreach ($ids as $id) {
        $this->ids[] = $id;
      }
    }

    // Set the ids to exclude from export.
    if ($options['exclude-ids'] !== 'none') {
      $exclude_ids = explode(',', $options['exclude-ids']);

      foreach ($exclude_ids as $exclude_id) {
        $this->excludeIds[] = $exclude_id;
      }
    }

    // Prepare for export.
    $this->initializeEntityExportFilters();
    $this->initializeFieldProcessors();
    $this->prepareExportDir();

    // Export each entity.
    foreach ($this->entityTypes as $entity_type) {
      $this->logger()->notice(dt('Gathering @entity content...', ['@entity' => $entity_type]));

      // Get all entities of the given type.
      $entity_storage = $this->entityTypeManager->getStorage($entity_type);
      $entity_keys = $entity_storage->getEntityType()->getKeys();
      $entities = [];
      $entity_query = $entity_storage->getQuery();
      $entity_query->accessCheck(FALSE);

      // If there are bundles specified, get only those entities.
      if (isset($this->bundles[$entity_type])) {
        $entity_query->condition($entity_keys['bundle'], $this->bundles[$entity_type], 'IN');
      }

      // If there are ids specified, get only those entities.
      if (!empty($this->ids)) {
        $entity_query->condition($entity_keys['id'], $this->ids, 'IN');
      }

      // If there are ids to exclude, remove them.
      if (!empty($this->excludeIds)) {
        $entity_query->condition($entity_keys['id'], $this->excludeIds, 'NOT IN');
      }

      $entity_ids = $entity_query->execute();

      if (!empty($entity_ids)) {
        $entities = $entity_storage->loadMultiple($entity_ids);
      }

      foreach ($entities as $entity) {
        // Pass the entity through any applicable filters.
        $check = TRUE;

        if (isset($this->entityExportFilters[$entity_type])) {
          foreach ($this->entityExportFilters[$entity_type] as $filter) {
            if (!$filter->filter($entity)) {
              $check = FALSE;
              break;
            }
          }
        }

        // If the entity didn't pass the filter, skip it.
        if (!$check) {
          continue;
        }

        // Get the bundle for the entity.
        $bundle = $entity->bundle();

        // Get the fields for the entity.
        $field_definitions = $this->entityFieldManager->getFieldDefinitions($entity_type, $bundle);

        /** @var \Drupal\Core\Entity\ContentEntityInterface $entity */
        $translations = $entity->getTranslationLanguages();

        foreach ($translations as $translation) {
          $translation_id = $translation->getId();
          $translation_entity = $entity->getTranslation($translation_id);

          // Create an array to hold the processes entity for export.
          $entry = [];

          // Loop through all the fields and pass them through
          // any applicable field processors.
          foreach ($field_definitions as $field_name => $field_definition) {
            // If the field is in the default ignored fields, skip it.
            if (in_array($field_name, self::DEFAULT_IGNORED_FIELDS)) {
              continue;
            }

            // Check for and set a cached processor for the field type.
            if (!isset($this->fieldProcessorCache[$field_name])) {
              foreach ($this->fieldProcessors as $processor) {
                if ($processor->isApplicable($translation_entity, $field_name, $field_definition)) {
                  $this->fieldProcessorCache[$field_name] = $processor;
                  break;
                }
              }
            }

            // Set the processed value for the field.
            $entry[$field_name] = $this->fieldProcessorCache[$field_name]->getValue($translation_entity, $field_name, $field_definition);
          }

          // Make sure the uuid field is moved to be first in the array.
          $entry = $this->prependUuid($entry);

          // Run the cleanup.
          $entry = $this->cleanupEntity($entry, $options);

          // Add the entity (as an array) to the results.
          if ($translation_entity->isDefaultTranslation()) {
            $this->content[$entity_type][$bundle]['_default'][$entry['uuid']] = $entry;
          }
          else {
            // Make sure the uuid is transformed into the translation origin
            // and a unique key is added to the array.
            $entry = $this->prepareTranslation($entry, $translation_id);

            $this->content[$entity_type][$bundle][$translation_id][$entry['translation_origin']] = $entry;
          }
        }
      }
    }

    // Write the content to the export directory.
    $this->writeContent();

    $this->logger()->success(dt('Export complete!'));
  }

  /**
   * Prepends the uuid to an entity array.
   *
   * @param array $entity
   *   The entity to alter.
   *
   * @return array
   *   The altered entity.
   */
  protected function prependUuid(array $entity): array {
    $uuid = $entity['uuid'];
    unset($entity['uuid']);
    return ['uuid' => $uuid] + $entity;
  }

  /**
   * Initializes all entity export filters for use.
   */
  protected function initializeEntityExportFilters() {
    $entity_export_filters = $this->exportEntityFilterPluginManager->createInstances();

    // Sort the filters into entity types.
    foreach ($entity_export_filters as $filter) {
      $entity_types = $filter->getEntityTypes();

      foreach ($entity_types as $entity_type) {
        $this->entityExportFilters[$entity_type][] = $filter;
      }
    }
  }

  /**
   * Initializes all field processors for use.
   */
  protected function initializeFieldProcessors() {
    // Get all the field processors.
    $this->fieldProcessors = $this->fieldProcessorPluginManager->createInstances();
    $generic_processor = NULL;

    // Find the generic field processor and ensure it is last in the array.
    foreach ($this->fieldProcessors as $index => $processor) {
      /** @var \Drupal\migrate_default_content_export\FieldProcessorBase $processor */
      if ($processor->getPluginId() === 'generic_field_processor') {
        $generic_processor = $processor;
        unset($this->fieldProcessors[$index]);
        break;
      }
    }

    if ($generic_processor !== NULL) {
      $this->fieldProcessors[] = $generic_processor;
    }
  }

  /**
   * Sets the export directory path and creates it if it doesn't exist.
   */
  protected function prepareExportDir() {
    // Get the root of the drupal site.
    $root = \Drupal::root();

    // Get the default_content source directory from config.
    $config = $this->configFactory->get('migrate_default_content.settings');
    $source_dir = $config->get('source_dir');

    // Get the default_content export directory from config.
    $config = $this->configFactory->get('migrate_default_content_export.settings');
    $export_dir = $config->get('export_dir');

    $this->exportDir = $root . '/' . $source_dir . '/' . $export_dir;

    if (!file_exists($this->exportDir)) {
      mkdir($this->exportDir, 0777, TRUE);
    }
  }

  /**
   * Processes an option string into an entity/bundle array.
   *
   * @param string $option
   *   The option string to process.
   *
   * @return array
   *   An array containing the entity and bundle.
   */
  protected function processOption(string $option) {
    $option_info = [
      'entity' => NULL,
      'bundle' => NULL,
    ];

    if (strpos($option, ':') !== FALSE) {
      $option = explode(':', $option);
      $option_info['entity'] = $option[0];
      $option_info['bundle'] = $option[1];
    }
    else {
      $option_info['entity'] = $option;
    }

    return $option_info;
  }

  /**
   * Validates that an entity type exists _and_ is a content entity.
   *
   * @param string $entity_type
   *   The entity type to validate.
   *
   * @return bool
   *   TRUE if the entity type exists and is content, FALSE otherwise.
   */
  protected function validateEntity($entity_type) {
    // First, make sure the entity type exists.
    if (!$this->entityTypeManager->hasDefinition($entity_type)) {
      $error = dt('Entity type "@entity" does not exist.', ['@entity' => $entity_type]);
      $this->errorMessage = $error;

      return FALSE;
    }

    // Then make sure the entity is content.
    $entity_type_definition = $this->entityTypeManager->getDefinition($entity_type);

    if ($entity_type_definition->getGroup() !== 'content') {
      $error = dt('Entity type "@entity" is not content.', ['@entity' => $entity_type]);
      $this->errorMessage = $error;

      return FALSE;
    }

    return TRUE;
  }

  /**
   * Validates an entity type is bundleable.
   *
   * @param string $entity_type
   *   The entity type to validate.
   *
   * @return bool
   *   TRUE if the entity type is bundleable, FALSE otherwise.
   */
  protected function validateBundleableEntity($entity_type) {
    // Then make sure the entity is content.
    $entity_type_definition = $this->entityTypeManager->getDefinition($entity_type);

    // Then make sure the entity type is bundleable.
    if ($entity_type_definition->getBundleEntityType() === NULL) {
      $error = dt('Entity type "@entity" is not bundleable.', ['@entity' => $entity_type]);
      $this->errorMessage = $error;

      return FALSE;
    }

    return TRUE;
  }

  /**
   * Validates an entity type has a specific bundle.
   *
   * @param string $entity_type
   *   The entity type to validate.
   * @param string $bundle
   *   The bundle to validate.
   *
   * @return bool
   *   TRUE if the entity type has the bundle, FALSE otherwise.
   */
  protected function validateBundle($entity_type, $bundle) {
    // Then make sure the bundle exists.
    $bundle_info = $this->entityTypeBundleInfo->getBundleInfo($entity_type);

    if (!isset($bundle_info[$bundle])) {
      $error = dt('Bundle "@bundle" does not exist for entity type "@entity".', [
        '@bundle' => $bundle,
        '@entity' => $entity_type,
      ]);
      $this->errorMessage = $error;

      return FALSE;
    }

    return TRUE;
  }

  /**
   * Adds an entity to the list of entities to export.
   *
   * @param string $entity_type
   *   The entity type to add.
   */
  protected function addEntity($entity_type) {
    $this->entityTypes[] = $entity_type;
  }

  /**
   * Removes an entity and it's bundles from the list of entities to export.
   *
   * @param string $entity_type
   *   The entity type to remove.
   */
  protected function removeEntity($entity_type) {
    $this->entityTypes = array_diff($this->entityTypes, [$entity_type]);
    $this->removeBundles($entity_type);
  }

  /**
   * Adds a bundle to the list of bundles to export.
   *
   * @param string $entity_type
   *   The entity type to add the bundle to.
   * @param string $bundle
   *   The bundle to add.
   */
  protected function addBundle($entity_type, $bundle) {
    if (!isset($this->bundles[$entity_type])) {
      $this->bundles[$entity_type] = [];
    }

    $this->bundles[$entity_type][] = $bundle;
  }

  /**
   * Removes a bundle from the list of bundles to export.
   *
   * @param string $entity_type
   *   The entity type to remove the bundle from.
   * @param string $bundle
   *   The bundle to remove.
   */
  protected function removeBundle($entity_type, $bundle) {
    if (isset($this->bundles[$entity_type])) {
      $this->bundles[$entity_type] = array_diff($this->bundles[$entity_type], [$bundle]);

      // Clean up the bundles array if it's empty.
      if (empty($this->bundles[$entity_type])) {
        unset($this->bundles[$entity_type]);
      }
    }
  }

  /**
   * Adds all of an entities bundles to the list of bundles to export.
   *
   * @param string $entity_type
   *   The entity type to add the bundles to.
   */
  protected function addBundles($entity_type) {
    $bundle_info = $this->entityTypeBundleInfo->getBundleInfo($entity_type);

    foreach ($bundle_info as $bundle => $info) {
      $this->addBundle($entity_type, $bundle);
    }
  }

  /**
   * Removes all of an entities bundles from the list of bundles to export.
   *
   * @param string $entity_type
   *   The entity type to remove the bundles from.
   */
  protected function removeBundles($entity_type) {
    if (isset($this->bundles[$entity_type])) {
      unset($this->bundles[$entity_type]);
    }
  }

  /**
   * Writes the content to the export directory.
   */
  protected function writeContent() {
    $this->logger()->notice(dt('Writing content to export directory...'));

    foreach ($this->content as $entity_type => $bundles) {
      foreach ($bundles as $bundle => $languages) {
        foreach ($languages as $language => $entities) {
          $filename = $entity_type . '.' . $bundle . '.yml';

          // If the language is not the default, add it to the filename.
          if ($language !== '_default') {
            $filename = $entity_type . '.' . $bundle . '.' . $language . '.yml';
          }

          $filepath = $this->exportDir . '/' . $filename;

          // Remove the keys from the entities.
          $entities = array_values($entities);

          // Convert the entities to yaml.
          $yaml = \Drupal::service('serialization.yaml')->encode($entities);

          // Write the yaml to a file.
          file_put_contents($filepath, $yaml);
        }
      }
    }
  }

  /**
   * Cleans up the an entity by transforming values.
   *
   * This will:
   *   - remove null values unless the allow-null option is set.
   *   - convert strings that are numbers to actual numbers.
   *
   * @param array $entity
   *   The entity to clean up.
   * @param array $options
   *   The options passed to the command.
   *
   * @return array
   *   The cleaned up entity.
   */
  protected function cleanupEntity(array $entity, array $options): array {
    foreach ($entity as $key => $value) {
      // Remove null values unless the option is set.
      if (!$options['allow-null'] && $value === NULL) {
        unset($entity[$key]);
      }

      // Convert numbers to actual numbers.
      if (is_numeric($value)) {
        $entity[$key] = (int) $value;
      }
    }

    return $entity;
  }

  /**
   * Prepends a key and translation origin to an entity array using the uuid.
   *
   * This also removes the uuid from the entity.
   *
   * @param array $entity
   *   The entity to alter.
   * @param string $langcode
   *   The language code of the translation.
   *
   * @return array
   *   The altered entity.
   */
  protected function prepareTranslation(array $entity, string $langcode): array {
    $uuid = $entity['uuid'];
    $translation_key = $langcode . '-' . $uuid;

    // Remove the uuid from the entity.
    unset($entity['uuid']);

    // Set an identifier array.
    $identifier = [
      'translation_key' => $translation_key,
      'translation_origin' => $translation_key,
    ];

    // Add the translation origin and a unique key to the entity.
    return $identifier + $entity;
  }

}
