<?php

declare(strict_types = 1);

namespace Drupal\migrate_default_content_export\Plugin\FieldProcessor;

use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\migrate_default_content_export\FieldProcessorPluginBase;

/**
 * An color field processor plugin.
 *
 * @FieldProcessor(
 *   id = "color_field_processor",
 *   label = @Translation("Color Field Processor"),
 *   description = @Translation("A field processor that returns the color values of the field.")
 * )
 */
class ColorFieldProcessor extends FieldProcessorPluginBase {

  /**
   * {@inheritdoc}
   */
  public function isApplicable(ContentEntityInterface $entity, string $field_name, FieldDefinitionInterface $field_definition): bool {
    // Color fields are applicable.
    return $field_definition->getType() === 'color_field_type';
  }

  /**
   * {@inheritdoc}
   */
  protected function process(mixed $value, FieldDefinitionInterface $field_definition): mixed {
    // If the value is empty, return NULL.
    if (empty($value) || !is_array($value)) {
      return NULL;
    }

    // Create a new array to store the processed values.
    $values = [];

    // Process each item in the value array.
    foreach ($value as $item) {
      if (empty($item['color'])) {
        $values[] = NULL;
      }
      else {
        $values[] = [
          'color' => $item['color'],
          'opacity' => $item['opacity'] ?? NULL,
        ];
      }
    }

    // If the processed values array is empty, return NULL.
    if (empty($values)) {
      return NULL;
    }

    return count($values) === 1 ? reset($values) : $values;
  }

}
