<?php

declare(strict_types = 1);

namespace Drupal\migrate_default_content_export\Plugin\FieldProcessor;

use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Field\FieldDefinitionInterface;

/**
 * A user reference field processor plugin.
 *
 * @FieldProcessor(
 *   id = "user_reference_field_processor",
 *   label = @Translation("User Reference Field Processor"),
 *   description = @Translation("A field processor that handles non-pregenerated users as entity references.")
 * )
 */
final class UserReferenceFieldProcessor extends EntityReferenceFieldProcessor {

  /**
   * Uids that should be ignored.
   */
  const IGNORE_UIDS = [
    0,
    1,
  ];

  /**
   * {@inheritdoc}
   */
  public function isApplicable(ContentEntityInterface $entity, string $field_name, FieldDefinitionInterface $field_definition): bool {
    // Entity reference and entity reference revisions fields that are
    // referencing users are applicable.
    $is_entity_reference = $field_definition->getType() === 'entity_reference';
    $is_entity_reference_revisions = $field_definition->getType() === 'entity_reference_revisions';

    // If the field is not an entity reference or entity reference revisions
    // field, return false.
    if (!($is_entity_reference || $is_entity_reference_revisions)) {
      return FALSE;
    }

    // Make sure the field is referencing users.
    $settings = $field_definition->getSettings();
    $is_user = !empty($settings['target_type']) && $settings['target_type'] === 'user';

    // If the field is not referencing users, return false.
    if (!$is_user) {
      return FALSE;
    }

    // Get the value of the field to check if it should be ignored.
    $value = parent::getValue($entity, $field_name, $field_definition);

    // If the value is in the ignore list, return false.
    if (in_array($value, self::IGNORE_UIDS)) {
      return FALSE;
    }

    return TRUE;
  }

}
