<?php

namespace Drupal\migrate_default_content\Drush\Commands;

use Drupal\Component\Serialization\Yaml;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\migrate_default_content\MigrationGeneratorInterface;
use Drush\Attributes as CLI;
use Drush\Commands\DrushCommands;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * A Drush commandfile.
 *
 * In addition to this file, you need a drush.services.yml
 * in root of your module, and a composer.json file that provides the name
 * of the services file to use.
 */
final class MigrateDefaultContentCommands extends DrushCommands {

  /**
   * The exported migrations.
   *
   * @var array
   */
  private $migrations;

  /**
   * The export directory.
   *
   * @var string
   */
  private $exportDir;

  /**
   * Constructs a MigrateDefaultContentCommands object.
   */
  public function __construct(
    private readonly MigrationGeneratorInterface $migrateDefaultContentMigrationGenerator,
    private readonly Yaml $yaml,
    private readonly ConfigFactoryInterface $configFactory,
  ) {
    parent::__construct();
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('migrate_default_content.migration_generator'),
      $container->get('serialization.yaml'),
      $container->get('config.factory'),
    );
  }

  /**
   * Generates and exports all migrations that migrate_default_content will use.
   */
  #[CLI\Command(name: 'migrate-default-content:export-migrations', aliases: ['mdcem'])]
  #[CLI\Usage(name: 'migrate-default-content:export-migrations', description: 'Exports all generated migrations.')]
  public function commandName() {
    // Generate the migrations.
    $this->migrations = $this->migrateDefaultContentMigrationGenerator->generateMigrations();

    // Prepare the export directory.
    $this->prepareExportDir();

    // Write the migrations to the export directory.
    $this->writeMigrations();
  }

  /**
   * Sets the export directory path and creates it if it doesn't exist.
   */
  protected function prepareExportDir() {
    // Get the root of the drupal site.
    $root = \Drupal::root();

    // Get the default_content source directory from config.
    $config = $this->configFactory->get('migrate_default_content.settings');
    $source_dir = $config->get('source_dir');

    // Get the migration export directory from config.
    $export_dir = $config->get('migration_export_dir');

    $this->exportDir = $root . '/' . $source_dir . '/' . $export_dir;

    if (!file_exists($this->exportDir)) {
      mkdir($this->exportDir, 0777, TRUE);
    }
  }

  /**
   * Writes the migrations to the export directory.
   */
  protected function writeMigrations() {
    foreach ($this->migrations as $migration) {
      $filename = $this->exportDir . '/' . $migration['id'] . '.yml';
      $yaml = $this->yaml->encode($migration);
      file_put_contents($filename, $yaml);
    }
  }

}
