<?php

namespace Drupal\migrate_devel\EventSubscriber;

use Drupal\migrate\Event\MigrateEvents;
use Drupal\migrate\Event\MigratePostRowSaveEvent;
use Drupal\migrate\Event\MigratePreRowSaveEvent;
use Drush\Drush;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\VarDumper\Cloner\VarCloner;
use Symfony\Component\VarDumper\Dumper\CliDumper;

/**
 * MigrationEventSubscriber for Debugging Migrations.
 *
 * @class MigrationEventSubscriber
 */
class MigrationEventSubscriber implements EventSubscriberInterface {

  /**
   * Pre Row Save Function for --migrate-debug-pre.
   *
   * @param \Drupal\migrate\Event\MigratePreRowSaveEvent $event
   *   Pre-Row-Save Migrate Event.
   */
  public function debugRowPreSave(MigratePreRowSaveEvent $event) {
    if (PHP_SAPI !== 'cli') {
      return;
    }

    $row = $event->getRow();

    if (in_array('migrate-debug-pre', Drush::config()->get('runtime.options'))) {
      // Start with capital letter for variables since this is actually a label.
      $source = $row->getSource();
      $destination = $row->getDestination();

      $this->printDebugOutput([
        '---------------------------------------------------------------------',
        '|                             Source                                |',
        '---------------------------------------------------------------------',
        $source,
        '---------------------------------------------------------------------',
        '|                           Destination                             |',
        '---------------------------------------------------------------------',
        $destination,
      ]);
    }
  }

  /**
   * Post Row Save Function for --migrate-debug.
   *
   * @param \Drupal\migrate\Event\MigratePostRowSaveEvent $event
   *   Post-Row-Save Migrate Event.
   */
  public function debugRowPostSave(MigratePostRowSaveEvent $event) {
    if (PHP_SAPI !== 'cli') {
      return;
    }

    $row = $event->getRow();

    if (in_array('migrate-debug', Drush::config()->get('runtime.options'))) {

      // Start with capital letter for variables since this is actually a label.
      $source = $row->getSource();
      $destination = $row->getDestination();
      $destinationIDValues = $event->getDestinationIdValues();

      $this->printDebugOutput([
        '---------------------------------------------------------------------',
        '|                             Source                                |',
        '---------------------------------------------------------------------',
        $source,
        '---------------------------------------------------------------------',
        '|                           Destination                             |',
        '---------------------------------------------------------------------',
        $destination,
        '---------------------------------------------------------------------',
        '|                       DestinationIdValues                         |',
        '---------------------------------------------------------------------',
        $destinationIDValues,
      ]);
    }
  }

  /**
   * Print debug output without line numbers.
   *
   * @param array $items
   *   Array of items to print. Strings are printed as-is, arrays are formatted.
   */
  protected function printDebugOutput(array $items) {
    $cloner = new VarCloner();
    $dumper = new CliDumper();
    $dumper->setColors(TRUE);
    $output = Drush::output();

    // Print a newline before the debug output to ensure it starts on a fresh
    // line, clearing any drush progress output that may be on the current line.
    $output->writeln('');

    foreach ($items as $item) {
      if (is_string($item)) {
        $output->writeln($item);
      }
      else {
        $dumper->dump(
          $cloner->cloneVar($item),
          function ($line, $depth, $indent_pad) use ($output) {
            // A negative depth means "end of dump".
            if ($depth >= 0) {
              // Output the line with indentation but without line numbers.
              $output->writeln(str_repeat($indent_pad, $depth) . $line);
            }
          }
        );
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    $events[MigrateEvents::PRE_ROW_SAVE][] = ['debugRowPreSave'];
    $events[MigrateEvents::POST_ROW_SAVE][] = ['debugRowPostSave'];
    return $events;
  }

}
