<?php

namespace Drupal\Tests\mile\Functional;

use Drupal\Core\Entity\EntityInterface;
use Drupal\menu_link_content\Entity\MenuLinkContent;
use Drupal\Tests\BrowserTestBase;

/**
 * Tests the mile referencing functionality.
 *
 * @group mile
 */
class MileEntityReferenceTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'block',
    'block_content',
    'node',
    'menu_link_content',
    'mile',
  ];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'olivero';

  /**
   * Tests MILE block references on save.
   */
  public function testMileBlockReference(): void {
    $this->login();
    $menu_link_item = $this->createMenuItemWithMileRefs('block', 'olivero_powered');

    $this->drupalGet($menu_link_item->toUrl('edit-form'));
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->fieldValueEquals('mile_references[mile_content_type]', 'block');
    $this->assertSession()->fieldValueEquals('mile_references[block_target]', 'Powered by Drupal (olivero_powered)');
  }

  /**
   * Tests MILE block_content references on save.
   */
  public function testMileBlockContentReference(): void {
    $this->login();
    $block_content = $this->createBlockContent();
    $menu_link_item = $this->createMenuItemWithMileRefs('block_content', $block_content->id());

    $this->drupalGet($menu_link_item->toUrl('edit-form'));
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->fieldValueEquals('mile_references[mile_content_type]', 'block_content');
    $this->assertSession()->fieldValueEquals('mile_references[block_content_target]', 'First (' . $block_content->id() . ')');
  }

  /**
   * Tests MILE node references on save.
   */
  public function testMileNodeReference(): void {
    $this->login();
    $this->drupalCreateContentType(['type' => 'article']);
    $node = $this->drupalCreateNode([
      'title' => t('MILE Test node'),
      'type' => 'article',
      'status' => TRUE,
    ]);
    $menu_link_item = $this->createMenuItemWithMileRefs('node', $node->id());

    $this->drupalGet($menu_link_item->toUrl('edit-form'));
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->fieldValueEquals('mile_references[mile_content_type]', 'node');
    $this->assertSession()->fieldValueEquals('mile_references[node_target]', 'MILE Test node (' . $node->id() . ')');
  }

  /**
   * Drupal login.
   */
  public function login($allow_mile_access = TRUE): void {
    $permissions[] = 'administer content types';
    $permissions[] = 'administer menu';
    if ($allow_mile_access) {
      $permissions[] = 'administer mile references';
    }

    $user = $this->drupalCreateUser($permissions);
    $this->drupalLogin($user);
  }

  /**
   * Create menu item.
   */
  public function createMenuItemWithMileRefs(string $entity_type, string $id): MenuLinkContent {
    $menu_link_item = MenuLinkContent::create([
      'title' => 'MILE link test',
      'provider' => 'menu_link_content',
      'menu_name' => 'main',
      'link' => [
        'uri' => 'internal:/node',
        'options' => [
          'mile_references' => [
            'mile_content_type' => $entity_type,
            $entity_type . '_target' => $this->loadEntity($entity_type, $id),
          ],
        ],
      ],
    ]);

    $menu_link_item->save();
    return $menu_link_item;
  }

  /**
   * Create an entity using ID.
   */
  public function loadEntity(string $entity_type, string $id) {
    return \Drupal::entityTypeManager()
      ->getStorage($entity_type)
      ->load($id);
  }

  /**
   * Create Block content.
   */
  public function createBlockContent(): EntityInterface {
    $first_block_content = \Drupal::entityTypeManager()
      ->getStorage('block_content')
      ->create([
        'type' => 'basic',
        'info' => 'First',
        'body' => [
          'value' => 'First test block content for MILE',
        ],
      ]
    );
    $first_block_content->save();
    return $first_block_content;
  }

}
