<?php

namespace Drupal\miniorange_2fa\Plugin\AuthenticationType;

use Drupal\miniorange_2fa\Annotation\AuthenticationType;
use Drupal\miniorange_2fa\Plugin\OtpAuthenticationTypePluginBase;

/**
 * @AuthenticationType(
 *   id = "otp-over-sms-and-email",
 *   name = "OTP Over SMS and EMAIL",
 *   code = "SMS AND EMAIL",
 *   type = "OTP",
 *   description = "You will receive the same OTP (One Time Passcode) via SMS and email. You have to enter the OTP to authenticate yourself.",
 *   supported_devices = {"Laptops", "Feature Phones", "Smartphones"},
 *   challenge = true,
 *   oob = false
 * )
 */
class SmsAndEmailAuthenticationType extends OtpAuthenticationTypePluginBase {

  /**
   * {@inheritdoc}
   */
  protected function shouldRenderPhoneField(): bool {
    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  protected function renderEmailField(string $email): array {
    return [
      '#type' => 'email',
      '#title' => $this->t('Email Address'),
      '#default_value' => $email,
      '#required' => TRUE,
      '#attributes' => ['id' => 'query_email', 'autocomplete' => 'on'],
      '#disabled' => TRUE,
    ];
  }

  /**
   * {@inheritdoc}
   */
  protected function addMethodSpecificWarnings(): void {
    $credits = $this->getRemainingCredits();
    $sms_remaining = $credits['mo_auth_2fa_sms_remaining'] ?? 0;
    $email_remaining = $credits['mo_auth_2fa_email_remaining'] ?? 0;
    
    if ($sms_remaining == 0) {
      $this->addWarningMessage('Zero SMS Transactions Remaining');
    }
    elseif ($sms_remaining <= 10) {
      $this->addWarningMessage('Low SMS Transactions Remaining: @count SMS left.', ['@count' => $sms_remaining]);
    }

    if ($email_remaining == 0) {
      $this->addWarningMessage('Zero Email Transactions Remaining');
    }
    elseif ($email_remaining <= 10) {
      $this->addWarningMessage('Low Email Transactions Remaining: @count emails left.', ['@count' => $email_remaining]);
    }
  }
}
