<?php

namespace Drupal\miniorange_2fa\Plugin\AuthenticationType;

use Drupal\miniorange_2fa\Plugin\AuthenticationTypePluginBase;
use Drupal\miniorange_2fa\Annotation\AuthenticationType;
use Drupal\miniorange_2fa\UsersAPIHandler;
use Drupal\Core\Form\FormStateInterface;
use Drupal\miniorange_2fa\MoAuthUtilities;
use Drupal\user\Entity\User;

/**
 * Yubikey Hardware Token authentication type plugin.
 *
 * @AuthenticationType(
 *   id = "hardware-token",
 *   name = @Translation("Yubikey Hardware Token"),
 *   code = "HARDWARE TOKEN",
 *   type = "OTHER",
 *   description = @Translation("Use a Yubikey hardware token for two-factor authentication."),
 *   supported_devices = {"desktop", "laptop"},
 *   challenge = true,
 *   oob = false,
 *   doc_link = "https://www.drupal.org/docs/extending-drupal/contributed-modules/contributed-modules/setup-guides-to-configure-various-2fa-mfa-tfa-methods/other-tfa-methods/setup-yubikey-hardware-token-as-2fa-tfa-method",
 *   video_link = ""
 * )
 */
class YubikeyHardwareTokenType extends AuthenticationTypePluginBase {

  /**
   * Maximum length for hardware token passcode.
   */
  const MAX_TOKEN_LENGTH = 60;

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $this->setFormTitle($form, 'hardware-token-methods');
    
    $form['mo_configure_hardware_token'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Hardware Token One Time Passcode'),
      '#description' => $this->t('Insert the Hardware Token in the USB Port and touch button on Hardware token.'),
      '#attributes' => [
        'placeholder' => $this->t('Enter the token'),
        'autofocus' => true,
        'autocomplete' => 'off',
      ],
      '#required' => true,
      '#maxlength' => self::MAX_TOKEN_LENGTH,
    ];

    $form['mo_hardware_token_action'] = [
      '#type' => 'actions',
    ];

    $form['mo_hardware_token_action']['done'] = [
      '#type' => 'submit',
      '#value' => $this->t('Submit'),
      '#button_type' => 'primary',
    ];
    
    $this->addFormSuffix($form);
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state): void {
   
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    $form_values = $form_state->getValues();
    $current_user = $this->getCurrentUser();

    if (!$this->validateUserSession()) {
      $this->handlePostAuthenticationRedirection(
        false,
        '',
        'User session validation failed.',
        $form_state
      );
      return;
    }
    
    $user = $this->getCurrentUser();
    if (!$user) {
      $this->handlePostAuthenticationRedirection(
        false,
        '',
        'User not found.',
        $form_state
      );
      return;
    }

    $user_id = $user->id();
    $session = MoAuthUtilities::getSession();
    $moMfaSession = $session->get('mo_auth');
    $user_email = $this->getUserEmail($user_id) ?: $moMfaSession['email'] ?? '';
    
    if (empty($user_email)) {
      $this->handlePostAuthenticationRedirection(
        false,
        '',
        'User email not found.',
        $form_state
      );
      return;
    }

    $hardware_token = trim($form_values['mo_configure_hardware_token']);
    
    $customer = $this->createCustomerProfile();
    $auth_api_handler = $this->createAuthApiHandler();
    $user_api_handler = new UsersAPIHandler($customer->getCustomerID(), $customer->getAPIKey());
    
    $miniorange_user = $this->createMiniorangeUser($this->getCode());
    if (!$miniorange_user) {
      $this->handlePostAuthenticationRedirection(
        false,
        '',
        'Failed to create user profile.',
        $form_state
      );
      return;
    }
    
    $response = $auth_api_handler->register($miniorange_user, $this->getCode(), null, $hardware_token, null);
    
   if(is_object($response) && $response->status == 'ERROR'){
   
    $this->handlePostAuthenticationRedirection(
      false,
      '',
      $response->message ?? 'An error occurred while processing your request.',
      $form_state
    );
    return;
   }

    $result = $this->updateUserAuthenticationMethod($this->getCode(), $miniorange_user, $user_api_handler);
    
    $this->handlePostAuthenticationRedirection(
      $result['success'],
      $result['message'],
      $result['message'],
      $form_state
    );
  }

} 