(function (Drupal, drupalSettings, once) {
  let floodTimerInterval = null;
  let blockTimerInterval = null;

  Drupal.behaviors.moFloodControl = {
    attach: function (context, settings) {
      const STORAGE_KEYS = {
        FLOOD: 'mo_flood_remaining',
        BLOCK: 'mo_block_remaining',
        RESEND_COUNT: 'mo_resend_count',
        LAST_CLICKED: 'mo_last_clicked',
      };
      const isRequestForm = settings?.mo_auth?.is_request_form || false;
      const floodDuration = settings?.mo_auth?.flood_duration || 0;
      const blockDuration = settings?.mo_auth?.block_duration || 0;
      const resendLimit = settings?.mo_auth?.resend_limit || 0;
      const resendCountBackend = settings?.mo_auth?.resend_count || 0;
      const floodTimerBackend = settings?.mo_auth?.flood_timer || 0;
      const blockTimerBackend = settings?.mo_auth?.block_timer || 0;

      const timerEl = document.getElementById('resend-timer');
      const requestBtn = document.querySelector('input[name="request_otp"]');
      const resendBtn = document.querySelector('input[name="resend_otp"]');
      const activeBtn = isRequestForm ? requestBtn : resendBtn;

      const actionLabel = isRequestForm ? 'request OTP' : 'resend OTP'; 

      const updateButtonState = (disabled) => {
        if (activeBtn) {
          activeBtn.disabled = disabled;
          activeBtn.classList.toggle('button--disabled', disabled);
          activeBtn.classList.toggle('is-disabled', disabled);
        }
      };

      const stopAllTimers = () => {
        clearInterval(floodTimerInterval);
        clearInterval(blockTimerInterval);
        floodTimerInterval = blockTimerInterval = null;
        localStorage.removeItem(STORAGE_KEYS.FLOOD);
        localStorage.removeItem(STORAGE_KEYS.BLOCK);
        if (timerEl) timerEl.style.display = 'none';
        updateButtonState(false);
      };

      const startTimer = (type, remainingSeconds) => {
        clearInterval(floodTimerInterval);
        clearInterval(blockTimerInterval);

        const isBlock = type === 'block';
        const storageKey = isBlock ? STORAGE_KEYS.BLOCK : STORAGE_KEYS.FLOOD;
        let remaining = remainingSeconds;

        updateButtonState(true);
        localStorage.setItem(storageKey, remaining);

        const messagePrefix = isBlock
          ? 'You have reached the maximum OTP resend limit.'
          : `You can ${actionLabel} after`; 

        if (timerEl) {
          timerEl.style.display = 'block';
          timerEl.innerText = `${messagePrefix} ${remaining} seconds.`;
        }

        const intervalId = setInterval(() => {
          remaining--;
          if (remaining > 0) {
            localStorage.setItem(storageKey, remaining);
            if (timerEl) {
              timerEl.innerText = `${messagePrefix} ${remaining} seconds.`;
            }
          } else {
            clearInterval(intervalId);
            localStorage.removeItem(storageKey);
            if (isBlock) localStorage.removeItem(STORAGE_KEYS.RESEND_COUNT);
            stopAllTimers();
          }
        }, 1000);

        if (isBlock) blockTimerInterval = intervalId;
        else floodTimerInterval = intervalId;
      };

      const getStoredTimer = () => {
        const storedBlock = localStorage.getItem(STORAGE_KEYS.BLOCK);
        const storedFlood = localStorage.getItem(STORAGE_KEYS.FLOOD);
        if (storedBlock && parseInt(storedBlock) > 0 && resendLimit > 0) {
          return { type: 'block', remaining: parseInt(storedBlock) };
        }
        if (storedFlood && parseInt(storedFlood) > 0 && floodDuration > 0) {
          return { type: 'flood', remaining: parseInt(storedFlood) };
        }
        return null;
      };

      const getBackendTimer = () => {
        if (blockTimerBackend > 0 && resendLimit > 0) {
          return { type: 'block', remaining: blockTimerBackend };
        }
        if (floodTimerBackend > 0 && floodDuration > 0) {
          return { type: 'flood', remaining: floodTimerBackend };
        }
        return null;
      };

      if (!resendLimit || resendLimit === 0) {
        localStorage.removeItem(STORAGE_KEYS.BLOCK);
        localStorage.removeItem(STORAGE_KEYS.RESEND_COUNT);
      }

      if (isRequestForm) {
        const storedTimer = getStoredTimer();
        if (storedTimer) {
          startTimer(storedTimer.type, storedTimer.remaining);
          return;
        }

        const backendTimer = getBackendTimer();
        if (backendTimer) {
          startTimer(backendTimer.type, backendTimer.remaining);
          return;
        }

        stopAllTimers();
        return;
      }

      let resendCount = parseInt(localStorage.getItem(STORAGE_KEYS.RESEND_COUNT) || resendCountBackend || 0);

      once('detectClickedButton', 'input.js-form-submit', context).forEach(function (button) {
        button.addEventListener('click', function (event) {
          const clicked = event.target.name;
          if (clicked === 'resend_otp') {
            resendCount++;
            localStorage.setItem(STORAGE_KEYS.RESEND_COUNT, resendCount);
          }
          localStorage.setItem(STORAGE_KEYS.LAST_CLICKED, clicked);
        });
      });

      const lastClicked = localStorage.getItem(STORAGE_KEYS.LAST_CLICKED);
      const storedTimer = getStoredTimer();

      if (storedTimer) {
        startTimer(storedTimer.type, storedTimer.remaining);
        return;
      }

      if (lastClicked === 'resend_otp') {
        localStorage.removeItem(STORAGE_KEYS.LAST_CLICKED);

        if (resendLimit && resendCount >= resendLimit) {
          startTimer('block', blockDuration || 120);
          localStorage.setItem(STORAGE_KEYS.RESEND_COUNT, 0);
          return;
        }

        if (floodDuration > 0) {
          startTimer('flood', floodDuration);
          return;
        }
      }

      stopAllTimers();
    }
  };
})(Drupal, drupalSettings, once);
