<?php

namespace Drupal\miniorange_2fa\Commands;

use Drush\Commands\DrushCommands;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Database\Connection;

/**
 * Provides Drush commands for enabling/disabling 2FA.
 */
class Miniorange2faDrushCommands extends DrushCommands {

  /**
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * @var \Drupal\Core\Database\Connection
   */
  protected $connection;

  /**
   * Constructor.
   */
  public function __construct(EntityTypeManagerInterface $entityTypeManager, Connection $connection) {
    parent::__construct();
    $this->entityTypeManager = $entityTypeManager;
    $this->connection = $connection;
  }

  /**
   * Enable or disable 2FA for an admin user.
   *
   * @command miniorange_2fa:change-status
   * @aliases mo-2fa-status
   *
   * @param string $action
   *   Action to perform: "enable" or "disable".
   * @param string $email
   *   Administrator's email address whose 2FA status will be changed.
   */
  public function changeTwoFactorStatus(string $action, string $email) {
    $action = strtolower($action);
  
    if (!in_array($action, ['enable', 'disable'])) {
      $this->io()->error('Invalid action. Use "enable" or "disable".');
      return;
    }
  
    $config = \Drupal::config('miniorange_2fa.settings');
    $is_drush_enabled = (int) $config->get('mo_auth_2fa_drush');
  
    if (!$is_drush_enabled) {
      $this->io()->error('Drush control for 2FA is disabled in configuration. Cannot proceed.');
      return;
    }
  
    $user_record = $this->connection->select('UserAuthenticationType', 'uat')
        ->fields('uat', ['uid', 'miniorange_registered_email', 'enabled'])
        ->condition('miniorange_registered_email', $email)
        ->execute()
        ->fetchAssoc();
  
    if (!$user_record) {
      $users = $this->entityTypeManager->getStorage('user')->loadByProperties(['mail' => $email]);
      $user = reset($users);
      
      if (!$user) {
        $this->io()->error("No user found with email: {$email}. User needs to configure 2FA first.");
        return;
      }
      
      $uid = $user->id();
      
      $user_record = $this->connection->select('UserAuthenticationType', 'uat')
          ->fields('uat', ['uid', 'enabled'])
          ->condition('uid', $uid)
          ->execute()
          ->fetchAssoc();
      
      if (!$user_record) {
        $this->io()->error("User '{$email}' (UID: {$uid}) does not have 2FA configured yet.");
        return;
      }
    } else {
      $uid = $user_record['uid'];
      $user = $this->entityTypeManager->getStorage('user')->load($uid);
      
      if (!$user) {
        $this->io()->error("User with UID: {$uid} not found in Drupal. Database inconsistency detected.");
        return;
      }
    }
  
    $current_status = $user_record['enabled'] == 1 ? 'enabled' : 'disabled';
    $this->io()->info("Current 2FA status for '{$email}': {$current_status}");
  
    if (!$user->hasPermission('administer site configuration')) {
      $this->io()->error("The user '{$user->getAccountName()}' lacks permission to manage site configuration.");
      return;
    }

    $enabled_value = ($action === 'enable') ? 0 : 1;
    $status_text = ($action === 'enable') ? 'enabled' : 'disabled';

    if (class_exists('\Drupal\miniorange_2fa\MoAuthUtilities')) {
      \Drupal\miniorange_2fa\MoAuthUtilities::update_user_status_from_UserAuthentication_table($uid, $enabled_value);
      $this->logger()->notice("2FA {$status_text} via Drush for admin {$email} (UID: {$uid}).");
      $this->io()->success("2FA has been {$status_text} successfully for '{$email}'.");
    } else {
      $this->io()->error("MoAuthUtilities class not found. Cannot update 2FA status.");
    }
  }

}
