<?php

/**
 * @file
 * Contains form for customer setup.
 */

namespace Drupal\miniorange_2fa\Form;

use Drupal\user\Entity\User;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\miniorange_2fa\MiniorangeUser;
use Drupal\miniorange_2fa\UsersAPIHandler;
use Drupal\miniorange_2fa\MoAuthUtilities;
use Drupal\miniorange_2fa\MoAuthConstants;
use Drupal\miniorange_2fa\AuthenticationType;
use Drupal\miniorange_2fa\MiniorangeCustomerSetup;
use Drupal\miniorange_2fa\MiniorangeCustomerProfile;
use Drupal\miniorange_2fa\Helper\FormHelper\MoAuthTitle;
use Drupal\miniorange_2fa\Services\LicenseManager;
use Drupal\Core\Database\Connection;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\miniorange_2fa\Helper\FormHelper\MoAuthCustomerSetupFormComponents;

/**
 * Customer setup form().
 */
class MoAuthCustomerSetup extends FormBase
{

  /**
   * The license manager service.
   *
   * @var \Drupal\miniorange_2fa\Services\LicenseManager
   */
  protected $licenseManager;

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * Constructor.
   *
   * @param \Drupal\miniorange_2fa\Services\LicenseManager $license_manager
   *   The license manager service.
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection service.
   */
  public function __construct(LicenseManager $license_manager, Connection $database)
  {
    $this->licenseManager = $license_manager;
    $this->database = $database;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container)
  {
    return new static(
      $container->get('miniorange_2fa.license_manager'),
      $container->get('database')
    );
  }

  public function getFormId()
  {
    return 'miniorange_2fa_customer_setup';
  }

  public function buildForm(array $form, FormStateInterface $form_state)
  {
    $user_obj = User::load(\Drupal::currentUser()->id());
    $user_id = $user_obj->id();
    $current_status = \Drupal::config('miniorange_2fa.settings')->get('mo_auth_status');

    $title = [
      'name'        => $this->t('Login'),
      'description' => $this->t('Login with your miniOrange account.'),
    ];

    MoAuthTitle::buildTitleForm($form, $form_state, $title);

    $form['markup_library'] = array(
      '#attached' => array(
        'library' => array(
          'miniorange_2fa/miniorange_2fa.license',
          "miniorange_2fa/miniorange_2fa.admin",
          "core/drupal.dialog.ajax",
        ),
      ),
    );

    if ($current_status == 'PLUGIN_CONFIGURATION') {
      return MoAuthCustomerSetupFormComponents::getPluginConfigurationForm($form, $form_state, $user_id);
    }

    $url = Url::fromRoute('miniorange_2fa.customer_setup')->toString();

    /**
     * Create container to hold @Login form elements.
     */
    return MoAuthCustomerSetupFormComponents::getLoginForm($form, $form_state, $url);
  }

  /**
   * Handle license fetch form submission.
   */
  public function fetchLicenseSubmit(array &$form, FormStateInterface $form_state)
  {
    $license_response = $this->licenseManager->fetchLicense();

    if ($license_response) {
      \Drupal::messenger()->addStatus(t('License information updated successfully.'));
    } else {
      \Drupal::messenger()->addError(t('Failed to fetch license information. Please try again or contact support.'));
    }
  }

  public function validateForm(array &$form, FormStateInterface $form_state)
  {
    // Validation logic removed as registration form is no longer available
  }

  //Handle submit for customer setup.
  public function submitForm(array &$form, FormStateInterface $form_state)
  {
    $check_loggers = MoAuthUtilities::get_mo_tab_url('LOGS');
    $user = User::load(\Drupal::currentUser()->id());
    $user_id = $user->id();

    $username = trim($form['mo_login_form']['Mo_auth_customer_login_username']['#value']);
    $password = trim($form['mo_login_form']['Mo_auth_customer_login_password']['#value']);
    $phone = '';

    $customer_config = new MiniorangeCustomerSetup($username, $phone, $password);
    $check_customer_response = $customer_config->checkCustomer();
    $utilities = new MoAuthUtilities();
    
    if (is_object($check_customer_response) && $check_customer_response->status == 'CUSTOMER_NOT_FOUND') {
      \Drupal::messenger()->addError(t('The account with username <strong>@username</strong> does not exist. Please create an account at <a href="https://www.miniorange.com/businessfreetrial" target="_blank">miniOrange</a>.', array('@username' => $username)));
      return;
    } elseif (is_object($check_customer_response) && $check_customer_response->status == 'SUCCESS') {
      // Customer exists. Retrieve keys.
      $customer_keys_response = $customer_config->getCustomerKeys();
      if (json_last_error() == JSON_ERROR_NONE) {
        $this->mo_auth_save_customer($user_id, $customer_keys_response, $username, $phone);
        \Drupal::messenger()->addStatus(t('Your account has been retrieved successfully.'));
      } else {
        \Drupal::messenger()->addError(t('Invalid credentials'));
        return;
      }
    } elseif (is_object($check_customer_response) && $check_customer_response->status == 'TRANSACTION_LIMIT_EXCEEDED') {
      MoAuthUtilities::mo_add_loggers_for_failures($check_customer_response->message, 'error');
      \Drupal::messenger()->addError(t('Failed to send an OTP. Please check your internet connection.') . ' <a href="' . $check_loggers . ' " target="_blank">' . t('Click here') . ' </a>' . t('for more details.'));
      return;
    } elseif (is_object($check_customer_response) && $check_customer_response->status == 'CURL_ERROR') {
      \Drupal::messenger()->addError(t('cURL is not enabled. Please enable cURL'));
      return;
    } else {
      MoAuthUtilities::mo_add_loggers_for_failures(isset($check_customer_response->message) ? $check_customer_response->message : '', 'error');
      \Drupal::messenger()->addError(t('Something went wrong, Please try again. Click <a href="' . $check_loggers . ' " target="_blank"> here </a> for more details.'));
      return;
    }
  }

  // Save MO Customer

  function mo_auth_save_customer($user_id, $json, $username, $phone)
  {

    $utilities = new MoAuthUtilities();
    $variables_and_values = array(
      'mo_auth_customer_admin_email' => $username,
      'mo_auth_customer_admin_phone' => $phone,
      'mo_auth_customer_id' => isset($json->id) ? $json->id : '',
      'mo_auth_customer_api_key' => isset($json->apiKey) ? $json->apiKey : '',
      'mo_auth_customer_token_key' => isset($json->token) ? $json->token : '',
      'mo_auth_customer_app_secret' => isset($json->appSecret) ? $json->appSecret : '',
    );
    $utilities->miniOrange_set_get_configurations($variables_and_values, 'SET');

    //Stores the user id of the user who activates the module.
    $MiniorangeUser = array(
      'mo_auth_firstuser_id' => \Drupal::currentUser()->id(),
    );
    $utilities->miniOrange_set_get_configurations($MiniorangeUser, 'SET');

    $auth_method = AuthenticationType::$EMAIL['code'] . ', ' . AuthenticationType::$EMAIL_VERIFICATION['code'];
    $available = $utilities::check_for_userID($user_id);
    $fields = array(
      'uid' => $user_id,
      'configured_auth_methods' => $auth_method,
      'miniorange_registered_email' => $username,
    );

    if ($available == FALSE) {
      $this->database->insert('UserAuthenticationType')->fields($fields)->execute();
    } elseif ($available == TRUE) {
      $this->database->update('UserAuthenticationType')->fields(['miniorange_registered_email' => $username])->condition('uid', $user_id, '=')->execute();
    }

    $utilities->miniOrange_set_get_configurations(array('mo_auth_status' => 'PLUGIN_CONFIGURATION'), 'SET');

    // Update the customer second factor to OTP Over Email in miniOrange
    $customer = new MiniorangeCustomerProfile();
    $miniorange_user = new MiniorangeUser($customer->getCustomerID(), $username, '', '', AuthenticationType::$EMAIL['code']);
    $user_api_handler = new UsersAPIHandler($customer->getCustomerID(), $customer->getAPIKey());
    $user_api_handler->update($miniorange_user);
    $license_response = $this->licenseManager->fetchLicense();

    $license_type = 'DRUPAL_2FA';
    $license_plan = 'DRUPAL_2FA';
    $no_of_users = 1;

    if (is_object($license_response) && $license_response->status == 'CURL_ERROR') {
      \Drupal::messenger()->addError(t('cURL is not enabled. Please enable cURL'));
      return;
    } elseif (is_object($license_response) && isset($license_response->licenseExpiry) && $utilities->license_expired($license_response->licenseExpiry) && $license_response->status == 'SUCCESS') {
      $license_type = $license_response->licenseType;
      /**Delete the OR part once all the Drupal 8 2FA customers shift on the Drupal 2FA plan.*/

      if ($license_type == MoAuthConstants::LICENSE_TYPE || $license_type == 'DRUPAL8_2FA_MODULE') {
        $license_plan = $license_response->licensePlan;
      }
      $no_of_users = $license_response->noOfUsers;
    }

    $mo_db_values = $utilities->miniOrange_set_get_configurations(array('mo_auth_enable_two_factor'), 'GET');

    $variables_and_values_2 = array(
      'mo_auth_2fa_license_type' => $license_type,
      'mo_auth_2fa_license_plan' => $license_plan,
      'mo_auth_2fa_license_no_of_users' => $no_of_users,
      'mo_auth_2fa_ivr_remaining' => isset($license_response->ivrRemaining) ? $license_response->ivrRemaining : '-',
      'mo_auth_2fa_sms_remaining' => isset($license_response->smsRemaining) ? $license_response->smsRemaining : '-',
      'mo_auth_2fa_email_remaining' => isset($license_response->emailRemaining) ? $license_response->emailRemaining : '-',
      'mo_auth_2fa_license_expiry' => isset($license_response->licenseExpiry) ? date('Y-M-d H:i:s', strtotime($license_response->licenseExpiry)) : '-',
      'mo_auth_2fa_support_expiry' => isset($license_response->supportExpiry) ? date('Y-M-d H:i:s', strtotime($license_response->supportExpiry)) : '-',
      'mo_auth_enable_two_factor' => $mo_db_values['mo_auth_enable_two_factor'] == '' ? TRUE : $mo_db_values['mo_auth_enable_two_factor'],
      'mo_auth_enforce_inline_registration' => $license_type == 'DRUPAL_2FA' ? FALSE : TRUE,
      'mo_auth_license_fetch_at' => time(),
    );
    $utilities->miniOrange_set_get_configurations($variables_and_values_2, 'SET');
  }

}
