<?php

namespace Drupal\miniorange_2fa\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\miniorange_2fa\Services\RecoveryCodeService;
use Drupal\Core\Session\AccountProxyInterface;

class RecoveryCodesForm extends FormBase {

  protected RecoveryCodeService $recoveryService;
  protected AccountProxyInterface $currentUser;

  public function __construct(RecoveryCodeService $recoveryService, AccountProxyInterface $current_user) {
    $this->recoveryService = $recoveryService;
    $this->currentUser = $current_user;
  }

  public static function create(ContainerInterface $container) {
    return new static($container->get('miniorange_2fa.recovery_code_service'), $container->get('current_user'));
  }

  public function getFormId() {
    return 'miniorange_2fa_recovery_form';
  }

  public function buildForm(array $form, FormStateInterface $form_state, $user = NULL) {
    $uid = (int) $this->currentUser->id();
    $plain_codes = $this->recoveryService->getPlainCodesFromTemp($uid);
    $recovery_mode = \Drupal::request()->query->get('recovery');

    if($recovery_mode) {
      
        $recovery_uid = (int) $user;

        $form['#title'] = 'Verify Recovery Code';

        $form['recovery_code'] = [
            '#type' => 'textfield',
            '#title' => $this->t('Enter Recovery Code'),
            '#required' => TRUE,
            '#description' => $this->t('Enter one of your unused recovery codes.'),
            '#attributes' => [
              'maxlength' => 19,
              'pattern' => '[0-9 ]+',
              'inputmode' => 'numeric',
              'autocomplete' => 'off',
            ],
          ];
    
          $form['submit'] = [
            '#type' => 'submit',
            '#value' => $this->t('Verify Code'),
            '#submit' => ['::verifyRecoveryCodeSubmit'],
          ];

          $form['back'] = [
            '#type' => 'link',
            '#title' => $this->t('Back'),
            '#url' => Url::fromRoute('miniorange_2fa.authenticate_user', ['user' => $recovery_uid]),
            '#attributes' => ['class' => ['button', 'secondary']],
          ];

          return $form;
    }
    if (empty($plain_codes)) {
      $this->messenger()->addMessage($this->t('No recovery codes to display. They are shown only immediately after generation.'), 'warning');
      $form['back'] = [
        '#type' => 'link',
        '#title' => $this->t('Back to account'),
        '#url' => Url::fromRoute('entity.user.canonical', ['user' => $uid]),
      ];
      return $form;
    }

    $items = [];
    foreach ($plain_codes as $c) {
      $items[] = ['#markup' => '<code>' . $this->t('@code', ['@code' => $c]) . '</code>'];
    }
    $form['instructions'] = [
      '#markup' => $this->t('<p>These are your recovery codes. Save them somewhere safe. Each code can be used once.</p>'),
    ];
    $form['codes'] = [
      '#theme' => 'item_list',
      '#items' => $plain_codes,
    ];

    $form['actions'] = ['#type' => 'actions'];
    $form['actions']['download'] = [
      '#type' => 'link',
      '#title' => $this->t('Download'),
      '#url' => Url::fromRoute('miniorange_2fa.recovery_codes_download', ['user' => $uid]),
      '#attributes' => ['class' => ['button']],
    ];
    $form['actions']['back'] = [
      '#type' => 'link',
      '#title' => $this->t('Back'),
      '#url' => Url::fromRoute('entity.user.canonical', ['user' => $uid]),
      '#attributes' => ['class' => ['button', 'secondary']],
    ];

    return $form;
  }

  public function submitForm(array &$form, FormStateInterface $form_state) {
  }

  public function verifyRecoveryCodeSubmit(array &$form, FormStateInterface $form_state) {
    $recovery_code = trim($form_state->getValue('recovery_code'));
    $user_id = \Drupal::routeMatch()->getParameter('user');

    if ($this->recoveryService->verifyAndConsumeRecoveryCode($user_id, $recovery_code)) {
      $user = \Drupal::entityTypeManager()->getStorage('user')->load($user_id);
      user_login_finalize($user);
      
      $session = \Drupal::service('session');
      $session->remove('mo_2fa_recovery_user');
      $this->messenger()->addStatus($this->t('Successfully logged in using recovery code.'));

      $form_state->setRedirect('entity.user.canonical', ['user' => $user_id]);
    } else {
    
      $this->messenger()->addError($this->t('Invalid recovery code. Please try again.'));
      $form_state->setRebuild();
    }
  }
}
