<?php

namespace Drupal\miniorange_2fa\Plugin;

use Drupal\Component\Plugin\PluginInspectionInterface;
use Drupal\Core\Form\FormStateInterface;

/**
 * Defines the interface for authentication type plugins.
 *
 * All authentication method plugins (SMS, TOTP, Email, etc.) must implement this interface.
 */
interface AuthenticationTypeInterface extends PluginInspectionInterface {

  /**
   * Get the human-readable name of the authentication method.
   *
   * @return string
   *   The name of the authentication method.
   */
  public function getName(): string;

  /**
   * Get the unique code/identifier for this authentication method.
   *
   * @return string
   *   The unique method code (e.g., 'SMS', 'EMAIL', 'TOTP').
   */
  public function getCode(): string;

  /**
   * Get the type/category of the authentication method.
   *
   * @return string
   *   The category type (e.g., 'OTP', 'TOTP', 'PUSH').
   */
  public function getType(): string;

  /**
   * Get the description of the authentication method.
   *
   * @return string
   *   A user-friendly description.
   */
  public function getDescription(): string;

  /**
   * Get a list of supported device types.
   *
   * @return array
   *   Array of device types (e.g., 'mobile', 'tablet').
   */
  public function getSupportedDevices(): array;

  /**
   * Whether this method requires user interaction (like entering a code).
   *
   * @return bool
   *   TRUE if challenge is required, FALSE otherwise.
   */
  public function requiresChallenge(): bool;

  /**
   * Whether this method uses out-of-band communication (like SMS or email).
   *
   * @return bool
   *   TRUE if out-of-band communication is used, FALSE otherwise.
   */
  public function isOutOfBand(): bool;

  /**
   * Get the iOS app store link.
   *
   * @return string
   *   The iOS app store URL.
   */
  public function getIosLink(): string;

  /**
   * Get the Android app store link.
   *
   * @return string
   *   The Android app store URL.
   */
  public function getAndroidLink(): string;

  /**
   * Convert plugin metadata into an associative array.
   *
   * @return array
   *   Metadata for legacy or external use.
   */
  public function toArray(): array;

  /**
   * Build the configuration form for this authentication method.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array
   *   The form array.
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array;

  /**
   * Validate the configuration form.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state): void;

  /**
   * Submit the configuration form.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void;
}
