<?php

namespace Drupal\miniorange_2fa\Services;

use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\TempStore\PrivateTempStoreFactory;
use Drupal\Core\Messenger\MessengerInterface;
use Psr\Log\LoggerInterface;

class RecoveryCodeService {

  protected AccountProxyInterface $currentUser;
  protected PrivateTempStoreFactory $tempStoreFactory;
  protected $userData;
  protected MessengerInterface $messenger;
  protected LoggerInterface $logger;

  public function __construct(AccountProxyInterface $current_user, PrivateTempStoreFactory $temp_store_factory, $user_data, MessengerInterface $messenger, LoggerInterface $logger) {
    $this->currentUser = $current_user;
    $this->tempStoreFactory = $temp_store_factory;
    $this->userData = $user_data;
    $this->messenger = $messenger;
    $this->logger = $logger;
  }

  /**
   * Generate $count random plain codes, store hashed in user.data, plain in tempstore.
   *
   * @return array Plain codes (for display).
   */
  public function generateRecoveryCodes(int $uid, int $count = 8): array {
    $plain = [];
    for ($i = 0; $i < $count; $i++) {
      
     $plain[] = $this->generateCode();
    }

    $hashed = array_map(function($c) {
      return password_hash($c, PASSWORD_DEFAULT);
    }, $plain);

    $this->userData->set('miniorange_2fa', (string) $uid, 'recovery_codes', $hashed);
    $this->userData->set('miniorange_2fa', (string) $uid, 'recovery_generated_at', \time());

    $store = $this->tempStoreFactory->get('miniorange_2fa');
    $store->set("plain_codes:$uid", $plain);

    return $plain;
  }

  protected function generateCode(): string {
    $digits = '';
    for ($i = 0; $i < 15 ; $i++) {
      $digits .= random_int(0, 9);
    }
    return trim(chunk_split($digits, 3, ' '));
  }

  /**
   * Retrieve plain codes from tempstore (one-time). Returns NULL if not present.
   */
  public function getPlainCodesFromTemp($uid): ?array {
    $store = $this->tempStoreFactory->get('miniorange_2fa');
    $val = $store->get("plain_codes:$uid");
    return $val ?: NULL;
  }

  /**
   * Remove plain codes from tempstore.
   */
  public function clearPlainCodes($uid): void {
    $store = $this->tempStoreFactory->get('miniorange_2fa');
    $store->delete("plain_codes:$uid");
  }

  /**
   * Verify a submitted recovery code for the given user. If valid, consume it (remove).
   * Returns TRUE if valid and consumed, FALSE otherwise.
   */
  public function verifyAndConsumeRecoveryCode(int $uid, string $candidate): bool {
    $hashes = $this->userData->get('miniorange_2fa', (string) $uid, 'recovery_codes') ?? [];
    foreach ($hashes as $index => $hash) {
      if (password_verify($candidate, $hash)) {
        unset($hashes[$index]);
        $hashes = array_values($hashes);
        $this->userData->set('miniorange_2fa', (string) $uid, 'recovery_codes', $hashes);
        $this->logger->info('User {uid} used a recovery code.', ['uid' => $uid]);
        return TRUE;
      }
    }
    return FALSE;
  }

}
