<?php

declare(strict_types=1);

namespace Drupal\miniorange_oauth_client\Form;

use Drupal\Core\Entity\EntityForm;
use Drupal\Core\Form\FormStateInterface;
use Drupal\miniorange_oauth_client\Entity\MoRoleMapping;
use Drupal\miniorange_oauth_client\MoConstant\MoModuleConstant;
use Drupal\miniorange_oauth_client\MoFeatures\MoUnoFeatures\MoUnoLicenseTierManager;
use Drupal\miniorange_oauth_client\MoFeatures\MoUnoFeatures\MoUnoTestConfigAttributeListBuilder;
use Drupal\miniorange_oauth_client\MoFeatures\MoUnoFeatures\MoUnoTutorial;
use Drupal\miniorange_oauth_client\MoLibrary\MoGhostInvoker;
use Drupal\miniorange_oauth_client\MoService\MoAjaxTables;
use Drupal\miniorange_oauth_client\MoService\MoLayoutBuilder;
use Drupal\user\Entity\Role;
use Drupal\user\RoleInterface;

/**
 * MiniOrange Role Mapping Entity form.
 */
final class MoRoleMappingForm extends EntityForm {

  use MoLayoutBuilder;
  use MoGhostInvoker;
  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {

    $form = parent::buildForm($form, $form_state);
    $disabled = \Drupal::service('miniorange_oauth_client.uno_module_subscription_library')->getModuleStatus($form);
    $current_lic_ver = $this->call([MoUnoLicenseTierManager::class, 'getLicenseVersion']);
    /** @var MoRoleMapping $entity */
    $entity = $this->getEntity();

    $form['entity_key'] = [
      '#type' => 'hidden',
      '#value' => $entity->id(),
    ];
    $form['is_enabled'] = [
      '#type' => 'hidden',
      '#value' => (int)!$disabled
    ];

    $form['enable_role_mapping'] = [
      '#type' => 'checkbox',
      '#tier' => MoModuleConstant::MO_STANDARD_VERSION_TIER,
      '#default_value' => $entity->getEnableRoleMapping(),
      '#title' => t('Enable Role Mapping'),
      '#states' => [
        'disabled' => [
          [':input[name = "is_enabled"]' => ['value' => '0']],
        ],
      ],
    ];

    $form['default_role'] = [
      '#type' => 'select',
      '#tier' => MoModuleConstant::MO_STANDARD_VERSION_TIER,
      '#title' => t('Select the default role for new users:'),
      '#options' => ['' => '-- Select --'] + self::getUserRoles(false, false),
      '#default_value' => $entity->getDefaultRole() ?? '',
      '#states' => [
        'disabled' => [
          [':input[name = "enable_role_mapping"]' => ['checked' => FALSE]],
          [':input[name = "is_enabled"]' => ['value' => '0']],
          ],
        ],
    ];

    $form['role_mapping_settings_frame'] = [
      '#type' => 'details',
      '#tier' => MoModuleConstant::MO_PREMIUM_VERSION_TIER,
      '#title' => t('Role Mapping Settings'),
      '#open' => TRUE,
      '#states' => [
        'disabled' => [
          [':input[name="enable_role_mapping"]' => ['checked' => FALSE]],
          [':input[name = "is_enabled"]' => ['value' => '0']],
        ],
      ],
    ];

    $form['role_mapping_settings_frame']['disable_role_update'] = [
      '#type' => 'checkbox',
      '#tier' => MoModuleConstant::MO_PREMIUM_VERSION_TIER,
      '#default_value' => $entity->getDisableRoleUpdate(),
      '#title' => t('Keep Existing Roles'),
      '#description' => t('Users will be left will their existing roles if nothing mapped below matches.'),
      '#states' => [
        'disabled' => [
          [':input[name="enable_role_mapping"]' => ['checked' => FALSE]],
          [':input[name = "is_enabled"]' => ['value' => '0']],
        ],
      ],
    ];

    $form['role_mapping_settings_frame']['role_attribute'] = [
      '#type' => 'textfield',
      '#tier' => MoModuleConstant::MO_PREMIUM_VERSION_TIER,
      '#title' => t('Role Attribute'),
      '#id' => 'text_field',
      '#default_value' => $entity->getRoleAttribute(),
      '#attributes' => ['placeholder' => 'Enter Role Attribute'],
      '#states' => [
        'disabled' => [
          [':input[name="enable_role_mapping"]' => ['checked' => FALSE]],
          [':input[name = "is_enabled"]' => ['value' => '0']],
        ],
      ],
    ];

    $role_names_array = [1 => self::getUserRoles(false, false), 2 => $entity->getCustomRoles()];
    $role_mapping_rows = $entity->getCustomRoles() ?? [];
    $role_mapping_fields = $this->RoleMappingFields();
    $role_unique_id_array = MoAjaxTables::getUniqueID($form_state->get('role_attribute_mapping_table_uid'), $role_mapping_rows);
    $form_state->set('role_attribute_mapping_table_uid', $role_unique_id_array);
    $role_mapping_header = $this->RoleMappingHeader();

    $form['role_mapping_settings_frame']['role_attr_map_frame'] = [
      '#type' => 'container',
      '#tier' => MoModuleConstant::MO_PREMIUM_VERSION_TIER,
      '#id' => 'role-attribute-mapping-table-wrapper',
      'custom_roles' => MoAjaxTables::generateTables(
        'role-attribute-mapping-table',
        $role_mapping_fields,
        $role_unique_id_array,
        $role_mapping_rows,
        $role_mapping_header,
        $role_names_array
      ),
      'add_row' => MoAjaxTables::generateAddButton(
        'Add',
        [$this, 'addRowNew'],
        [$this, 'ajaxCallback'],
        'role-attribute-mapping-table-wrapper',
        '',
        $disabled || !in_array($current_lic_ver, MoModuleConstant::MO_PREMIUM_VERSION_TIER)
      ),
      '#states' => [
        'disabled' => [
          [':input[name="enable_role_mapping"]' => ['checked' => FALSE]],
          [':input[name = "is_enabled"]' => ['value' => '0']],
        ],
      ],
    ];

    if (!$entity->isNew()) {
      $form['actions']['delete'] = [
        '#type' => 'submit',
        '#value' => $this->t('Reset'),
        "#weight" => 2,
        '#disabled' => $disabled || !in_array($current_lic_ver, MoModuleConstant::MO_STANDARD_VERSION_TIER),
        '#attributes' => ['class' => ['button', 'button--danger']],
        '#submit' => ['::delete']
      ];
    }

    $form['actions']['submit']['#attributes']['class'] = ['button', 'button--primary'];
    $form['actions']['submit']['#weight'] = 1;
    $form['actions']['submit']['#disabled'] = $disabled || !in_array($current_lic_ver, MoModuleConstant::MO_STANDARD_VERSION_TIER);


    $result['mo_outer_container'] = [
      "#type" => "container",
      "#attributes" => ['class' => ['mo-outer-container']]
    ];
    $this->modifyFormTitleOnTier($form);

    $result['mo_outer_container'] += $this->buildMOLayout($form);

    $result['mo_outer_container']['tutorial_frame'] = self::getTutorialLayout();

    $this->call(
      [MoUnoTutorial::class, 'buildVideoTutorialFrame'],
      ['result' => &$result['mo_outer_container']['tutorial_frame']]
    );

    $this->call(
      [MoUnoTestConfigAttributeListBuilder::class, 'buildTestConfigurationFrame'],
      ['result' => &$result['mo_outer_container']['tutorial_frame'], 'entity_id' => $entity->id()]
    );

    return $result;
  }
  public function RoleMappingFields(): array {
    return [
      'drupal_role' => [
        'type' => 'select',
      ],
      'oauth_server_role' => [
        'type' => 'textfield',
        'placeholder' => 'Please enter the semicolon (;) seperated values'
      ],
      'delete_button_role' => [
        'type' => 'submit',
        'submit' => [$this, 'removeRow'],
        'callback' => [$this, 'ajaxCallback'],
        'wrapper' => 'role-attribute-mapping-table-wrapper',
        'value' => t('Delete'),
      ],
    ];
  }

  public function RoleMappingHeader(): array {
    return [
      ['data' => t('Drupal Role')],
      ['data' => t('OAuth Server/Provider Role')],
      ['data' => t('Action')],
    ];
  }

  public function addRowNew(array &$form, FormStateInterface $form_state)
  {
    $add_button = ($form_state->getTriggeringElement()['#name']);
    $rows = $form_state->getValue('total_rows_' . $add_button);
    $unique_array_id = $form_state->get('role_attribute_mapping_table_uid');
    for ($count = 1; $count <= $rows; $count++) {
      $uuid_service = \Drupal::service('uuid');
      $uuid = $uuid_service->generate();
      $unique_array_id[] = $uuid;
    }
    $form_state->set('role_attribute_mapping_table_uid', $unique_array_id);
    $form_state->setRebuild();
  }

  public function removeRow(array &$form, FormStateInterface $form_state)
  {
    $id = $form_state->getTriggeringElement()['#name'];
    $var_value = 'role_attribute_mapping_table_uid';
    $unique_id = $form_state->get($var_value);
    $unique_id = array_diff($unique_id, [$id]);
    if (empty($unique_id)) {
      $uuid_service = \Drupal::service('uuid');
      $unique_id[0] = $uuid_service->generate();
    }
    $form_state->set($var_value, $unique_id);
    $form_state->setRebuild();
  }


  public function ajaxCallback(array &$form, FormStateInterface $form_state) {
    return $form['mo_outer_container']['mo_outline']['role_mapping_settings_frame']['role_attr_map_frame'];
  }

  public function submitForm(&$form, FormStateInterface $form_state): void
  {
    $form_state->setRedirect(
      'mo_configuration.edit', ['mo_client_config' => $this->entity->id(), 'tab' => 'role_mapping']
    );
    parent::submitForm($form, $form_state);
    \Drupal::messenger()->addMessage("Mapped roles are saved successfully.");
  }
  public static function getUserRoles($anonymous = FALSE, $authenticated = FALSE, $permission = NULL): array {
    $allRoles = Role::loadMultiple();
    $roles = [];

    if (!$anonymous) {
      unset($allRoles[RoleInterface::ANONYMOUS_ID]);
    }

    if (!$authenticated) {
      unset($allRoles[RoleInterface::AUTHENTICATED_ID]);
    }

    if (!empty($permission)) {
      $allRoles = array_filter($allRoles, function ($role) use ($permission) {
        return $role->hasPermission($permission);
      });
    }

    foreach ($allRoles as $role) {
      $roles[$role->id()] = $role->label();
    }

    return $roles;
  }

  public function delete(array $form, FormStateInterface $form_state): void
  {
    try {
      $this->entity->delete();
      \Drupal::messenger()->addMessage("Mapped roles are removed successfully.");
    } catch (\Exception $exception) {
      //todo need to add catch block
    }
  }
}
