<?php

declare(strict_types=1);

namespace Drupal\miniorange_oauth_client\Form;

use Drupal\Core\Entity\EntityForm;
use Drupal\Core\Form\FormStateInterface;
use Drupal\miniorange_oauth_client\Entity\MoClientSettings;
use Drupal\miniorange_oauth_client\MoConstant\MoModuleConstant;
use Drupal\miniorange_oauth_client\MoFeatures\MoUnoFeatures\MoUnoLicenseTierManager;
use Drupal\miniorange_oauth_client\MoFeatures\MoUnoFeatures\MoUnoTestConfigAttributeListBuilder;
use Drupal\miniorange_oauth_client\MoFeatures\MoUnoFeatures\MoUnoTutorial;
use Drupal\miniorange_oauth_client\MoLibrary\MoGhostInvoker;
use Drupal\miniorange_oauth_client\MoService\MoLayoutBuilder;
use Drupal\Core\Url;

/**
 * MiniOrange Client Settings Entity form.
 */
final class MoClientSettingsForm extends EntityForm
{

  use MoGhostInvoker;
  use MoLayoutBuilder;

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array
  {
    $form = parent::buildForm($form, $form_state);
    $disabled = \Drupal::service('miniorange_oauth_client.uno_module_subscription_library')->getModuleStatus($form);

    /** @var MoClientSettings $entity */
    $entity = $this->getEntity();

    $form['entity_key'] = [
      '#type' => 'hidden',
      '#value' => $entity->id(),
    ];

    $form['client_settings_is_enabled'] = [
      '#type' => 'hidden',
      '#value' => (int)!$disabled
    ];

    $form['auto_create_user'] = [
      '#type' => 'checkbox',
      '#tier' => MoModuleConstant::MO_STANDARD_VERSION_TIER,
      '#default_value' => $entity->getAutoCreateUser(),
      '#title' => t('Enable auto creation of users'),
      '#description' => t('Users will be auto-created in Drupal on first SSO login (if they don\'t already have an account).'),
      '#states' => [
        'disabled' => [
          [':input[name="client_settings_is_enabled"]' => ['value' => '0']],
        ]
      ],
    ];

    $form['redirect_unregister'] = [
      '#type' => 'url',
      '#tier' => MoModuleConstant::MO_ENTERPRISE_VERSION_TIER,
      '#title' => t('Redirect URL for Unregistered Users'),
      '#attributes' => ['placeholder' => 'Enter complete URL'],
      '#default_value' => $entity->getRedirectUnregister(),
      '#description' => t('If empty, the un-registered users will be redirected to same page from where they initiated the login.'),
      '#states' => [
        'visible' => [
          [
            ':input[name = "auto_create_user"]' => ['checked' => False],
          ],
        ],
        'disabled' => [
          [':input[name="client_settings_is_enabled"]' => ['value' => '0']],
        ]
      ],
    ];

    $form['disable_new_user'] = [
      '#type' => 'checkbox',
      '#tier' => MoModuleConstant::MO_ENTERPRISE_VERSION_TIER,
      '#disabled' => $disabled,
      '#default_value' => $entity->getDisableNewUser(),
      '#title' => t('Create new users in Blocked Status'),
      '#states' => [
        'visible' => [
          [':input[name = "auto_create_user"]' => ['checked' => true]],
        ],
        'disabled' => [
          [':input[name="client_settings_is_enabled"]' => ['value' => '0']],
        ]
      ],
    ];

    $form['disabled_user_redirect_url'] = [
      '#type' => 'url',
      '#tier' => MoModuleConstant::MO_ENTERPRISE_VERSION_TIER,
      '#title' => t('Redirect URL for Blocked Users'),
      '#attributes' => ['placeholder' => 'Enter complete URL'],
      '#default_value' => $entity->getDisabledUserRedirectUrl(),
      '#description' => t('If empty, users will be redirected to same page from where they initiated the login.'),
      '#states' => [
        'visible' => [
          ':input[name="disable_new_user"]' => ['checked' => true],
          'and',
          ':input[name="auto_create_user"]' => ['checked' => true],
        ],
        'disabled' => [
          [':input[name="client_settings_is_enabled"]' => ['value' => '0']],
        ]
      ],
    ];

    $form['redirect_url_login_logout_frame'] = [
      '#type' => 'details',
      '#title' => t('Redirection after SSO login and logout'),
      '#open' => TRUE,
      '#states' => [
        'disabled' => [
          [':input[name="client_settings_is_enabled"]' => ['value' => '0']],
        ]
      ],
    ];

    $form['redirect_url_login_logout_frame']['anonymous_user_redirect_url'] = [
      '#type' => 'url',
      '#tier' => MoModuleConstant::MO_ENTERPRISE_VERSION_TIER,
      '#title' => t('Redirect URL for Anonymous User'),
      '#attributes' => ['placeholder' => 'Enter complete URL'],
      '#default_value' => $entity->getAnonymousUserRedirectUrl(),
      '#description' => t('Users will be redirected to this URL if the SSO login fails or is canceled.'),
      '#states' => [
        'disabled' => [
          [':input[name="client_settings_is_enabled"]' => ['value' => '0']],
        ]
      ],
    ];

    $form['redirect_url_login_logout_frame']['login_redirect_url'] = [
      '#type' => 'url',
      '#tier' => MoModuleConstant::MO_STANDARD_VERSION_TIER,
      '#title' => t('Redirect URL after login'),
      '#default_value' => $entity->getLoginRedirectUrl(),
      '#attributes' => ['placeholder' => 'Enter complete URL'],
      '#description' => t('Users will be taken to this URL after performing SSO.'),
      '#states' => [
        'disabled' => [
          [':input[name="client_settings_is_enabled"]' => ['value' => '0']],
        ]
      ],
    ];

    $form['redirect_url_login_logout_frame']['same_page_after_logout'] = [
      '#type' => 'checkbox',
      '#tier' => MoModuleConstant::MO_ENTERPRISE_VERSION_TIER,
      '#default_value' => $entity->getSamePageAfterLogout(),
      '#title' => t('Keep users on the same page after logout'),
      '#states' => [
        'disabled' => [
          [':input[name="client_settings_is_enabled"]' => ['value' => '0']],
        ]
      ],
    ];

    $form['redirect_url_login_logout_frame']['logout_redirect_url'] = [
      '#type' => 'url',
      '#tier' => MoModuleConstant::MO_STANDARD_VERSION_TIER,
      '#disabled' => $disabled,
      '#title' => t('Redirect URL after logout'),
      '#default_value' => $entity->getLogoutRedirectUrl(),
      '#attributes' => ['placeholder' => 'Enter complete URL'],
      '#description' => t('Users will be taken to this URL after performing logout.'),
      '#states' => [
        'disabled' => [
          [':input[name = "same_page_after_logout"]' => ['checked' => true]],
          [':input[name="client_settings_is_enabled"]' => ['value' => '0']],
        ]
      ],
    ];

    // Attribute/Role Based Restriction
    $form['attribute_role_base_restriction'] = [
      '#type' => 'details',
      '#tier' => MoModuleConstant::MO_ENTERPRISE_VERSION_TIER,
      '#title' => t('Attribute Based Restriction'),
      '#open' => TRUE,
      '#states' => [
        'disabled' => [
          [':input[name="client_settings_is_enabled"]' => ['value' => '0']],
        ]
      ],
    ];

    $form['attribute_role_base_restriction']['role_based_restriction'] = [
      '#type' => 'checkbox',
      '#tier' => MoModuleConstant::MO_ENTERPRISE_VERSION_TIER,
      '#title' => t('Enable Attribute Based Restriction'),
      '#default_value' => $entity->getRoleBasedRestriction() ?? 0,
      '#states' => [
        'disabled' => [
          [':input[name="client_settings_is_enabled"]' => ['value' => '0']],
        ]
      ],
    ];

    $form['attribute_role_base_restriction']['role_restriction_attrname'] = [
      '#type' => 'textfield',
      '#tier' => MoModuleConstant::MO_ENTERPRISE_VERSION_TIER,
      '#title' => t('Attribute Name'),
      '#default_value' => $entity->getRoleRestrictionAttrName() ?? '',
      '#description' => '<b>Note:</b> Please fill this <b>Attribute Name field</b> to enable Attribute based restriction. Users will be restricted based on this attribute coming from OAuth Provider',
      '#states' => [
        'required' => [
          ':input[name="role_based_restriction"]' => ['checked' => true],],
        'visible' => [
          ':input[name="role_based_restriction"]' => ['checked' => TRUE],
        ],
        'disabled' => [
          [':input[name="client_settings_is_enabled"]' => ['value' => '0']],
        ]
      ],
    ];

    $form['attribute_role_base_restriction']['restriction_type_frame'] = [
      '#type' => 'details',
      '#tier' => MoModuleConstant::MO_ENTERPRISE_VERSION_TIER,
      '#title' => t('Select the restriction type'),
      '#open' => TRUE,
      '#states' => [
        'visible' => [
          ':input[name="role_based_restriction"]' => ['checked' => TRUE],
        ],
        'disabled' => [
          [':input[name="client_settings_is_enabled"]' => ['value' => '0']],
        ]
      ]
    ];

    $form['attribute_role_base_restriction']['restriction_type_frame']['typeof_role_restriction'] = [
      '#type' => 'radios',
      '#tier' => MoModuleConstant::MO_ENTERPRISE_VERSION_TIER,
      '#options' => ['whitelist' => 'Allow below Attributes', 'blacklist' => 'Restrict below Attributes'],
      '#default_value' => $entity->getTypeofRoleRestriction() ?? 'whitelist',
      '#states' => [
        'disabled' => [
          [':input[name="client_settings_is_enabled"]' => ['value' => '0']],
        ]
      ],
    ];

    $form['attribute_role_base_restriction']['list_of_restricted_roles'] = [
      '#type' => 'textarea',
      '#tier' => MoModuleConstant::MO_ENTERPRISE_VERSION_TIER,
      '#title' => t('Enter the attribute values'),
      '#placeholder' => t('Enter the list of semicolon (;) separated Attribute values coming from OAuth Provider to validate.'),
      '#description' => t('Enter the list of semicolon (;) separated Attribute values coming from OAuth Provider to validate.'),
      '#default_value' => $entity->getListOfRestrictedRoles() ?? '',
      '#states' => [
        'required' => [
          ':input[name="role_based_restriction"]' => ['checked' => true],
        ],
        'visible' => [
          ':input[name="role_based_restriction"]' => ['checked' => TRUE],
        ],
        'disabled' => [
          [':input[name="client_settings_is_enabled"]' => ['value' => '0']],
        ]
      ],
    ];

    $form['attribute_role_base_restriction']['redirect_url_for_role_restriction'] = [
      '#type' => 'url',
      '#tier' => MoModuleConstant::MO_ENTERPRISE_VERSION_TIER,
      '#maxlength' => 512,
      '#title' => t('Redirect URL for Restricted Users'),
      '#description' => t('If empty, users will be redirected to the login page (<i>' . Url::fromRoute('user.login')->toString() . '</i>).'),
      '#default_value' => $entity->getRedirectUrlForRoleRestriction() ?? '',
      '#attributes' => [
        'placeholder' => 'Enter the URL on which you wish to Redirect the restricted users',
      ],
      '#states' => [
        'visible' => [
          ':input[name="role_based_restriction"]' => ['checked' => TRUE],
        ],
        'disabled' => [
          [':input[name="client_settings_is_enabled"]' => ['value' => '0']],
        ]
      ]
    ];

    $form['slo_frame'] = [
      '#type' => 'details',
      '#tier' => MoModuleConstant::MO_PREMIUM_VERSION_TIER,
      '#title' => t('Single log out'),
      '#open' => TRUE,
      '#states' => [
        'disabled' => [
          [':input[name="client_settings_is_enabled"]' => ['value' => '0']],
        ]
      ],
    ];

    $form['slo_frame']['enable_slo'] = [
      '#type' => 'checkbox',
      '#tier' => MoModuleConstant::MO_PREMIUM_VERSION_TIER,
      '#title' => t('Enable single log out'),
      '#default_value' => $entity->getEnableSlo(),
      '#description' => t('Automatically destroys session in your IDP on logout.'),
      '#states' => [
        'disabled' => [
          [':input[name="client_settings_is_enabled"]' => ['value' => '0']],
        ]
      ],
    ];

    $form['slo_frame']['single_logout_url'] = [
      '#type' => 'url',
      '#tier' => MoModuleConstant::MO_PREMIUM_VERSION_TIER,
      '#title' => t('IDP End Session Endpoint'),
      '#default_value' => $entity->getSingleLogoutUrl(),
      '#attributes' => ['placeholder' => 'Enter complete URL'],
      '#description' => t('URL to log out user from the Identity Provider when they log out from the Drupal site.'),
      '#states' => [
        'invisible' => [
          [':input[name="enable_slo"]' => ['checked' => FALSE]],
        ],
        'disabled' => [
          [':input[name="client_settings_is_enabled"]' => ['value' => '0']],
        ]
      ],
    ];

    $form['slo_frame']['idp_logout_redirect_uri_param'] = [
      '#type' => 'textfield',
      '#tier' => MoModuleConstant::MO_PREMIUM_VERSION_TIER,
      '#title' => t('Redirect URI Param Name of Identity Provider'),
      '#default_value' => $entity->getIdpLogoutRedirectUriParam(),
      '#states' => [
        'invisible' => [
          [':input[name="single_logout_url"]' => ['value' => '']],
          [':input[name="enable_slo"]' => ['checked' => FALSE]],
        ],
        'disabled' => [
          [':input[name="client_settings_is_enabled"]' => ['value' => '0']],
        ]
      ],
    ];

    $form['slo_frame']['include_id_token_in_slo'] = [
      '#type' => 'checkbox',
      '#tier' => MoModuleConstant::MO_PREMIUM_VERSION_TIER,
      '#title' => t('Include ID Token in End Session Endpoint'),
      '#default_value' => $entity->getIncludeIdTokenInSlo(),
      '#states' => [
        'invisible' => [
          [':input[name="single_logout_url"]' => ['value' => '']],
          [':input[name="enable_slo"]' => ['checked' => FALSE]],
        ],
        'disabled' => [
          [':input[name="client_settings_is_enabled"]' => ['value' => '0']],
        ]
      ],
    ];

    // add additional properties to the base form submit operation
    if (!$entity->isNew()) {
      $form['actions']['delete'] = [
        '#type' => 'submit',
        '#tier' => MoModuleConstant::MO_STANDARD_VERSION_TIER,
        '#value' => $this->t('Reset'),
        "#weight" => 2,
        '#attributes' => ['class' => ['button', 'button--danger']],
        '#submit' => ['::delete'],
        '#states' => [
          'disabled' => [
            [':input[name="client_settings_is_enabled"]' => ['value' => '0']],
          ]
        ],
      ];
    }

    $form['actions']['submit']['#attributes']['class'] = ['button', 'button--primary'];
    $form['actions']['submit']['#weight'] = 1;
    $form['actions']['submit']['#states']['disabled'][] = [':input[name="client_settings_is_enabled"]' => ['value' => '0']];
    $form['actions']['submit']['#tier'] = MoModuleConstant::MO_STANDARD_VERSION_TIER;

    $result['mo_outer_container'] = [
      "#type" => "container",
      "#attributes" => ['class' => ['mo-outer-container']]
    ];

    $this->modifyFormTitleOnTier($form);
    $result['mo_outer_container'] += $this->buildMOLayout($form);

    $result['mo_outer_container']['tutorial_frame'] = self::getTutorialLayout();

    $this->call(
      [MoUnoTutorial::class, 'buildVideoTutorialFrame'],
      ['result' => &$result['mo_outer_container']['tutorial_frame']]
    );
    $this->call(
      [MoUnoTestConfigAttributeListBuilder::class, 'buildTestConfigurationFrame'],
      ['result' => &$result['mo_outer_container']['tutorial_frame'], 'entity_id' => $entity->id()]
    );

    return $result;
  }

  public function validateForm(array &$form, FormStateInterface $form_state)
  {
    //todo need to perform validations for the URl based on the exact need on TIERS
    parent::validateForm($form, $form_state);
  }

  public function submitForm(&$form, FormStateInterface $form_state): void
  {
    $form_state->setRedirect(
      'mo_configuration.edit', ['mo_client_config' => $this->entity->id(), 'tab' => 'client_setting']
    );
    parent::submitForm($form, $form_state);
    \Drupal::messenger()->addStatus(t('Client settings saved successfully.'));
  }

  public function delete(array $form, FormStateInterface $form_state): void
  {
    try {
      $this->entity->delete();
      \Drupal::messenger()->addStatus(t('Client settings have been deleted successfully.'));
    } catch (\Exception $exception) {
      //todo need to add catch block
    }
  }
}
