<?php
/**
 * This file helps in building the cache components of the stored module data.
 *
 * This also has a registered service in the services.yml file
 *  #### mo_cache_library:
 *  #### class: Drupal\miniorange_oauth_client\MoLibrary\MoCacheLibrary
 *  #### arguments:
 *  #### - '@cache.default'
 *  #### - '@config.factory'
 *
 * @author miniOrange
 * @version 1.0
 */
namespace Drupal\miniorange_oauth_client\MoLibrary;

use Drupal\Core\Cache\CacheBackendInterface;

final class MoCacheLibrary
{

  /**
   * Build the cache by fetching from the database
   *
   * @param string $cid cache-id
   * @param CacheBackendInterface $cache_factory cache factory
   *
   * @return array
   * @throws \Exception
   */
  private static function buildConfigCache(
    string $cid,
    CacheBackendInterface $cache_factory,
  ):array {
    try {
      $config_factory = \Drupal::service('config.factory');
      $config = $config_factory->get($cid)->getRawData();

      // Store in cache with an expiration time
      $cache_factory->set($cid, $config, CacheBackendInterface::CACHE_PERMANENT);
      MoLogger::info(t('Cache against - \'').$cid.t('\' was successfully created.'));
      return $config;
    } catch (\Exception $e) {
      MoLogger::error(t('Error in creating the cache against - \':cid\'', [':cid'=>$cid]));
      throw $e;
    }
  }

  /**
   * This function is used to build cache for the entity data
   *
   * @param string $cid cache-id
   * @param mixed $cache_factory cache factory
   * @param string $entity_id Config entity ID
   *
   * @return mixed
   * @throws \Exception
   */
  private static function buildEntityCache(string $cid, mixed $cache_factory, string $entity_id): mixed
  {
    try {
      $factory = \Drupal::service('entity_type.manager')->getStorage($entity_id);
      $config = $factory->load($cid);

      // Store in cache with an expiration time
      $cache_factory->set($entity_id.$cid, $config, CacheBackendInterface::CACHE_PERMANENT);
      MoLogger::info(t('Cache against - \'').$entity_id.$cid.t('\' was successfully created.'));
      return $config;
    } catch (\Exception $e) {
      MoLogger::error(t('Error in creating the cache against - \':cid\'', [':cid'=>$entity_id.$cid]));
      throw $e;
    }
  }

  /**
   * Read the stored cache
   *
   * @param string $cid cache-id
   * @param bool $load_from_config
   * @param bool $load_from_entity
   * @param string|null $entity_id
   *
   * @return array
   * @throws \Exception
   */
  public static function readCache(
    string $cid,
    bool $load_from_config = true,
    bool $load_from_entity = false,
    ?string $entity_id = ''
  ):array {
    try {
      $cache_factory = \Drupal::service('cache.default');

      if (!$load_from_config && $load_from_entity) {
        // Check if data exists in cache
        if ($cache = $cache_factory->get($entity_id.$cid)) {
          return $cache->data;
        }
        return self::buildEntityCache($cid, $cache_factory, $entity_id);
      }
      // Check if data exists in cache
      if ($cache = $cache_factory->get($cid)) {
        return $cache->data;
      }
      return self::buildConfigCache($cid, $cache_factory);
    } catch (\Exception $e) {
      MoLogger::error('Problem in reading cache build against');
      throw $e;
    }
  }

  /**
   * Remove the cache build and stored against a particular cache-id
   *
   * @param string $cid cache-id
   *
   * @return void
   * @throws \Exception
   */
  public static function removeCache(string $cid): void
  {
    $cache_factory = \Drupal::service('cache.default');
    try {
      $cache_factory->delete($cid);
      MoLogger::info(t('Cache stored against - \'').$cid.t('\' was successfully removed.'));
    } catch (\Exception $e) {
      MoLogger::error(t('Issue in removing the cache stored against - \'').$cid.t('\'.'). var_export($e, TRUE));
      throw $e;
    }
  }
}
