<?php
declare(strict_types=1);

namespace Drupal\miniorange_oauth_client\MoLibrary;

use Error;
use Exception;
use ReflectionMethod;
/**
 * Trait GhostInvoker
 *
 * This trait provides functionality for dynamically invoking methods
 * on a class while ensuring that the method exists and its parameters
 * are validated before execution.
 *
 * Features:
 * - Dynamically calls class methods using an array callback.
 * - Validates method parameters against expected reflection types.
 * - Logs errors and throws exceptions in test mode.
 *
 * Usage:
 * - This trait can be included in any class to enable dynamic method calls.
 *
 * Access Guide:
 * - The only accessible method is this class is "call"
 * - Request structure
 * -- self::call(['class_name', 'method_name'] = [], parameters = [])
 * -- 'class_name' -> Should be strictly of string with complete namespace, !avoid using objects
 * -- 'method_name' -> Should be strictly of type string
 * -- 'parameters' -> Should be strictly of type array
 * --- example:
 * --- function($a, $b) //if a function can accept these two params the value passed should be ['a'=> "xyz", 'b' => "lmn"]
 */
trait MoGhostInvoker
{
  /**
   * Determine whether the site is in test mode (true = throws errors, false = silent fail)
   * @var bool
   */
  private static bool $site_in_test_mode = false;

  /**
   * Calls a method dynamically if it exists and matches the parameter count.
   *
   * @param array $callback [Class, Method]
   * @param array $args [parameters]
   *
   * @return mixed|null Result of the method call, or null if invalid.
   * @throws Exception
   */
  public function call(array $callback = [], array $args = []): mixed
  {
    [$class_name, $method] = $callback;
    $loader = require DRUPAL_ROOT . '/../vendor/autoload.php';

    if (!$loader) {
      throw new Exception("Please point the autoloader to run the library");
    }

    // Validate class & method existence
    if (
      !($file = $loader->findFile($class_name)) || !file_exists($file) || !class_exists($class_name) ||
      !method_exists($class_name, $method)
    ) {
      if (self::$site_in_test_mode) {
        MoLogger::notice("Class or method not found: $class_name::$method");
      }
      return null;
    }
    // validate the passed params if the site is in test mode
    if (self::$site_in_test_mode) {
      $reflection = new ReflectionMethod($class_name, $method);
      self::validateParams($class_name, $method, $args, $reflection);
    }
    $reflection = new ReflectionMethod($class_name, $method);
    // Handle static method call
    if ($reflection->isStatic()) {
      return call_user_func_array($callback, $args);
    }
    // Handle dependency injection for non-static method
    try {
      if (\Drupal::hasContainer()) {
        $instance = \Drupal::service('class_resolver')->getInstanceFromDefinition($class_name);
      }
    } catch (\Exception $e) {
      throw new \Exception("Service for $class_name not found. Ensure it's registered in services.yml.");
    }
    if (!isset($instance)) {
      throw new \Exception("Could not create instance for class '$class_name'.");
    }
    return call_user_func_array([$instance, $method], $args);
  }

  /**
   * Validates method parameters by checking their presence and types against the expected function signature.
   *
   * @param string $class_name The name of the class containing the method.
   * @param string $method_name The name of the method being validated.
   * @param array $actual_params The actual parameters passed to the method.
   * @param ReflectionMethod $reflection The Reflection of the method.
   *
   * @throws Exception If a required parameter is missing or if a parameter has an incorrect type or position.
   */
  private static function validateParams(
    string $class_name,
    string $method_name,
    array $actual_params,
    ReflectionMethod $reflection
  ): void {

    $reflection_params = $reflection->getParameters();

    $actual_param_keys = array_keys($actual_params);
    if (
      !empty(
      $un_necessary_params = array_diff(
        $actual_param_keys, array_map(fn (\ReflectionParameter $param) => $param->getName(), $reflection_params))
      )) {
      throw new Error(
        'Un-necessary parameters are passed: ("' . implode('", "', $un_necessary_params). '")
        to the function "'.$method_name.'" in class "'.$class_name.'".'
      );
    }

    foreach ($reflection_params as $reflection_param) {

      $position = $reflection_param->getPosition();
      $param_name = $reflection_param->getName();
      $param_type = $reflection_param->getType();
      $is_optional = $reflection_param->isOptional();
      $allow_null = $reflection_param->allowsNull();

      $actual_param_name = $actual_param_keys[$position] ?? null;
      $actual_param_value = $actual_param_name !== null ? ($actual_params[$actual_param_name] ?? null) : null;
      $actual_param_type = $actual_param_value !== null ?
        (gettype($actual_param_value) == 'object' ? get_class($actual_param_value) : gettype($actual_param_value)) :
        null;

      if (!$allow_null && !$is_optional && !isset($actual_params[$param_name])) {
        throw new Exception(
          "The parameter '$param_name' is required in function '$method_name' in class '$class_name'."
        );
      }

      if ($actual_param_name !== null && isset($actual_params[$param_name])) {
        self::checkPositionAndType(
          $actual_param_name, $param_name, $method_name, $class_name, $param_type, $actual_param_type, $actual_params[$param_name]
        );
      }
    }
  }

  /**
   * Checks if the actual parameter's position and type match the expected definition.
   *
   * @param string $actual_param_name The name of the actual parameter passed.
   * @param string $param_name The expected parameter name.
   * @param string $method_name The name of the function being validated.
   * @param string $class_name The name of the class containing the method.
   * @param \ReflectionType|null $param_type The expected type of the parameter.
   * @param string|null $actual_param_type The actual type of the passed parameter.
   * @param mixed $actual_param_value The actual value of the passed parameter
   *
   * @throws Exception If the parameter position is incorrect or if the type does not match.
   */
  private static function checkPositionAndType(
    string $actual_param_name,
    string $param_name,
    string $method_name,
    string $class_name,
    ?\ReflectionType $param_type,
    ?string $actual_param_type,
    mixed $actual_param_value
  ): void {
    if ($actual_param_name !== $param_name) {
      throw new Exception(
        "Position of parameter '$param_name' is mismatched in function '$method_name' in class '$class_name'."
      );
    }
    if ($param_type instanceof \ReflectionNamedType && $param_type->getName() !== $actual_param_type && !self::checkCompatibility($actual_param_value, $param_type->getName())) {
      throw new Exception(
        "The expected type for parameter ('$param_name') is '{$param_type->getName()}', but received '$actual_param_type'."
      );
    }
  }

  /**
   * Checks if a value is compatible with the given type.
   *
   * @param mixed $value The value to check.
   * @param string $expected_type The expected type (e.g., 'bool', 'int', 'MyClass').
   *
   * @return bool True if the value is compatible with the expected type.
   * @throws \ReflectionException
   */
  public static function checkCompatibility(mixed $value, string $expected_type): bool {

    $expected_type = strtolower($expected_type);
    // Handle scalar types
    if (is_scalar($value) || is_null($value)) {
      return match ($expected_type) {
        'bool', 'boolean' => is_bool($value),
        'int', 'integer' => is_int($value),
        'float', 'double' => is_float($value),
        'string' => is_string($value),
        'array' => is_array($value),
        'null' => is_null($value),
        default => false,
      };
    }

    // Handle objects
    if (is_object($value)) {
      return (new \ReflectionClass($expected_type))->isInstance($value);
    }
    return false;
  }
}
