<?php

declare(strict_types=1);

namespace Drupal\miniorange_oauth_client\Entity;

use Drupal\Core\Config\Entity\ConfigEntityBase;
use Drupal\Core\Entity\EntityMalformedException;
use Drupal\Core\Entity\EntityStorageException;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\miniorange_oauth_client\MoConstant\MoModuleConstant;
use Drupal\miniorange_oauth_client\MoFeatures\MoUnoFeatures\MoUnoLicenseTierManager;
use Drupal\miniorange_oauth_client\MoHelper\MoUtilities;
use Drupal\miniorange_oauth_client\MoLibrary\MoCacheLibrary;
use Drupal\miniorange_oauth_client\MoLibrary\MoGhostInvoker;
use Drupal\miniorange_oauth_client\MoFeatures\MoUnoFeatures\MoLicense\MoDTO\MoUnoLicense;

/**
 * Defines the miniOrange client configuration entity type.
 *
 * @ConfigEntityType(
 *   id = "mo_client_config",
 *   label = @Translation("miniOrange Client Configuration Entity"),
 *   label_collection = @Translation("miniOrange Client Configuration Entities"),
 *   label_singular = @Translation("Configuration for "),
 *   label_plural = @Translation("applications configured"),
 *   label_count = @PluralTranslation(
 *     singular = "@count miniOrange Client Configuration Entity",
 *     plural = "@count miniOrange Client Configuration Entities",
 *   ),
 *   handlers = {
 *     "list_builder" = "Drupal\miniorange_oauth_client\MoEntityListBuilder\MoClientConfigurationListBuilder",
 *     "form" = {
 *       "add" = "Drupal\miniorange_oauth_client\Form\MoClientConfigurationForm",
 *       "edit" = "Drupal\miniorange_oauth_client\Form\MoClientConfigurationForm",
 *       "delete" = "Drupal\Core\Entity\EntityDeleteForm",
 *     },
 *   },
 *   config_prefix = "mo_client_config",
 *   admin_permission = "mo_administrator",
 *   links = {
 *     "collection" = "/admin/config/people/mo-oauth-client/mo-client-config",
 *     "add-form" = "/admin/config/people/mo-oauth-client/mo-client-config/add",
 *     "edit-form" = "/admin/config/people/mo-oauth-client/mo-client-config/{mo_client_config}/edit",
 *     "delete-form" = "/admin/config/people/mo-oauth-client/mo-client-config/{mo_client_config}/delete",
 *   },
 *   entity_keys = {
 *     "id" = "id",
 *     "label" = "app_name"
 *   },
 *   config_export = {
 *     "id",
 *     "client_app",
 *     "app_name",
 *     "callback_url",
 *     "client_id",
 *     "grant_type",
 *     "client_secret",
 *     "scope",
 *     "login_protocol",
 *     "authorize_endpoint",
 *     "access_token_endpoint",
 *     "userinfo_endpoint",
 *     "group_info_endpoint",
 *     "jkws_endpoint",
 *     "authentication_method",
 *     "enable_login_with_oauth",
 *     "display_login_link",
 *     "login_link_text",
 *     "server_test_attributes",
 *     "server_test_attributeKeys"
 *   },
 * )
 */
final class MoClientConfiguration extends ConfigEntityBase {

  use MoGhostInvoker;

  protected string $id;
  protected ?string $client_app = '';
  protected ?string $app_name = '';
  protected ?string $callback_url = '';
  protected ?string $client_id  = '';
  protected ?string $grant_type = 'authorization_code';
  protected ?string $client_secret = '';
  protected ?string $scope = '';
  protected ?string $login_protocol = 'OAuth';
  protected ?string $authorize_endpoint = '';
  protected ?string $access_token_endpoint = '';
  protected ?string $userinfo_endpoint = '';
  protected ?string $group_info_endpoint = '';
  protected ?string $jkws_endpoint = '';
  protected ?string $login_link_text = '';
  protected ?string $authentication_method = 'values_in_header';
  protected ?bool $enable_login_with_oauth = true;
  protected ?bool $display_login_link = true;
  protected ?array $server_test_attributes = [];
  protected ?array $server_test_attributeKeys = [];

  public static function getEntityId(): string
  {
    return 'mo_client_config';
  }

  public static function loadActive(bool $allow_password_grant = true, ?bool $show_login_link = null): array
  {
    $configs = self::loadMultiple();
    /** @var self $config */
    foreach ($configs as $config) {
      if (
        !$config->getEnableLoginWithOauth() ||
        (!$allow_password_grant && $config->getGrantType() == 'password') ||
        (!is_null($show_login_link) && ($show_login_link !== $config->getDisplayLoginLink()))
      ) {
        unset($configs[$config->id()]);
      }
    }
    return $configs;
  }

  public static function loadPasswordGrants(bool $enable_login = true): array
  {
    $configs = self::loadMultiple();
    /** @var MoClientConfiguration $config */
    foreach ($configs as $config) {
      if ($config->getGrantType() !== 'password' || $enable_login !== $config->getEnableLoginWithOauth()) {
        unset($configs[$config->id()]);
      }
    }
    return $configs;
  }

  public static function removeCache(?string $entity_id)
  {
    if (empty($entity_id)) {
      return;
    }
    if ($entity = \Drupal::entityTypeManager()->getStorage(self::getEntityId())->load($entity_id)) {
      \Drupal::service('cache_tags.invalidator')->invalidateTags($entity->getCacheTags());
    }
  }

  public static function removeTestConfigAttributes(string $config_id): void
  {
    self::load($config_id)->setServerTestAttributes(null)->save();
    \Drupal::messenger()->addMessage('Test configuration attributes cleared successfully.');
  }

    public function preSave(EntityStorageInterface $storage): void
  {
    if ((($this->call([MoUnoLicense::class, 'readMe'])?->getNoOfSp() ?? 1) <= count($storage->loadMultiple())) && $this->isNew()) {
      throw new EntityMalformedException("The module client's configuration limit exceeds.");
    }
    parent::preSave($storage);
  }

  /**
   * @throws EntityStorageException
   */
  public static function preDelete(EntityStorageInterface $storage, array $entities): void
  {
    foreach ($entities as $entity) {
      MoAttributeMapping::load($entity->id())?->delete();
      MoGroupMapping::load($entity->id())?->delete();
      MoProfileMapping::load($entity->id())?->delete();
      MoRoleMapping::load($entity->id())?->delete();
      array_map(fn ($obj) => $obj->delete(), MoLoginReport::loadByProperties(['application' => $entity->id()]));
    }
    parent::preDelete($storage, $entities);
  }
  public function getClientApp(): ?string
  {
    return $this->client_app;
  }

  public function setClientApp(string $client_app): MoClientConfiguration
  {
    $this->client_app = $client_app;
    return $this;
  }

  public function getAppName(): ?string
  {
    return $this->app_name;
  }

  public function setAppName(string $app_name): MoClientConfiguration
  {
    $this->app_name = $app_name;
    return $this;
  }

  public function getCallbackUrl(): ?string
  {
    return $this->callback_url;
  }

  public function setCallbackUrl(string $callback_url): MoClientConfiguration
  {
    $this->callback_url = $callback_url;
    return $this;
  }

  public function getClientId(): ?string
  {
    return $this->client_id;
  }

  public function setClientId(string $client_id): MoClientConfiguration
  {
    $this->client_id = $client_id;
    return $this;
  }

  public function getGrantType(): ?string
  {
    return $this->grant_type;
  }

  public function setGrantType(string $grant_type): MoClientConfiguration
  {
    $this->grant_type = $grant_type;
    return $this;
  }

  public function getClientSecret(): ?string
  {
    if (empty($this->client_secret)) {
      return $this->client_secret;
    }
    return MoUtilities::decryptData($this->client_secret, $this->getEncryptionKey());
  }

  public function setClientSecret(?string $client_secret): MoClientConfiguration
  {
    if (!empty($client_secret)) {
      $this->client_secret = MoUtilities::encryptData($client_secret, $this->getEncryptionKey());
    } else {
      $this->client_secret = $client_secret;
    }
    return $this;
  }

  public function getScope(): ?string
  {
    return $this->scope;
  }

  public function setScope(string $scope): MoClientConfiguration
  {
    $this->scope = $scope;
    return $this;
  }

  public function getLoginProtocol(): ?string
  {
    return $this->login_protocol;
  }

  public function setLoginProtocol(string $login_protocol): MoClientConfiguration
  {
    $this->login_protocol = $login_protocol;
    return $this;
  }

  public function getAuthorizeEndpoint(): ?string
  {
    return $this->authorize_endpoint;
  }

  public function setAuthorizeEndpoint(string $authorize_endpoint): MoClientConfiguration
  {
    $this->authorize_endpoint = $authorize_endpoint;
    return $this;
  }

  public function getAccessTokenEndpoint(): ?string
  {
    return $this->access_token_endpoint;
  }

  public function setAccessTokenEndpoint(string $access_token_endpoint): MoClientConfiguration
  {
    $this->access_token_endpoint = $access_token_endpoint;
    return $this;
  }

  public function getUserinfoEndpoint(): ?string
  {
    return $this->userinfo_endpoint;
  }

  public function setUserinfoEndpoint(string $userinfo_endpoint): MoClientConfiguration
  {
    $this->userinfo_endpoint = $userinfo_endpoint;
    return $this;
  }

  public function getGroupInfoEndpoint(): ?string
  {
    return $this->group_info_endpoint;
  }

  public function setGroupInfoEndpoint(string $group_info_endpoint): MoClientConfiguration
  {
    $this->group_info_endpoint = $group_info_endpoint;
    return $this;
  }

  public function getJkwsEndpoint(): ?string
  {
    return $this->jkws_endpoint;
  }

  public function setJkwsEndpoint(string $jkws_endpoint): MoClientConfiguration
  {
    $this->jkws_endpoint = $jkws_endpoint;
    return $this;
  }

  public function getLoginLinkText(): ?string
  {
    return $this->login_link_text;
  }

  public function setLoginLinkText(string $login_link_text): MoClientConfiguration
  {
    $this->login_link_text = $login_link_text;
    return $this;
  }

  public function getAuthenticationMethod(): ?string
  {
    return $this->authentication_method;
  }

  public function setAuthenticationMethod(string $authentication_method): MoClientConfiguration
  {
    $this->authentication_method = $authentication_method;
    return $this;
  }

  public function getEnableLoginWithOauth(): ?bool
  {
    return $this->enable_login_with_oauth;
  }

  public function setEnableLoginWithOauth(bool $enable_login_with_oauth): MoClientConfiguration
  {
    $this->enable_login_with_oauth = $enable_login_with_oauth;
    return $this;
  }

  public function getDisplayLoginLink(): ?bool
  {
    return $this->display_login_link;
  }

  public function setDisplayLoginLink(bool $display_login_link): MoClientConfiguration
  {
    $this->display_login_link = $display_login_link;
    return $this;
  }

  // todo need to add a part to display all the server fetched attributes in UI
  public function getServerTestAttributes(): ?array
  {
    return $this->server_test_attributes;
  }

  public function setServerTestAttributes(?array $server_test_attributes): MoClientConfiguration
  {
    $this->server_test_attributes = $server_test_attributes;
    if ($server_test_attributes) {
      $this->setServerTestAttributeKeys(array_keys($server_test_attributes));
    }
    return $this;
  }

  public function getServerTestAttributeKeys(): ?array
  {
    return $this->server_test_attributeKeys;
  }

  public function setServerTestAttributeKeys(?array $server_test_attributeKeys): MoClientConfiguration
  {
    $this->server_test_attributeKeys = array_combine($server_test_attributeKeys, $server_test_attributeKeys);
    return $this;
  }

  private function getEncryptionKey()
  {
    return $this->call([MoUnoLicenseTierManager::class, 'getLicenseVersion']) == MoModuleConstant::MO_FREE_VERSION ?
      \Drupal::service('private_key')->get() : ((\Drupal::service('miniorange_oauth_client.uno_module_license_entity')->readMe())?->getCustomerToken() ?? '');
  }
}
