<?php

declare(strict_types=1);

namespace Drupal\miniorange_oauth_client\MoEntityListBuilder;

use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\OpenModalDialogCommand;
use Drupal\Core\Ajax\RedirectCommand;
use Drupal\Core\Config\Entity\ConfigEntityListBuilder;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Link;
use Drupal\Core\Url;
use Drupal\miniorange_oauth_client\Entity\MoClientConfiguration;
use Drupal\miniorange_oauth_client\MoFeatures\MoUnoFeatures\MoLicense\MoDTO\MoUnoLicense;
use Drupal\miniorange_oauth_client\MoHelper\MoClientConfigurationHelper;
use Drupal\miniorange_oauth_client\MoHelper\MoSupport;
use Drupal\miniorange_oauth_client\MoLibrary\MoGhostInvoker;
use Drupal\miniorange_oauth_client\MoService\MoLayoutBuilder;

/**
 * Provides a listing of miniorange client configuration entities.
 */
final class MoClientConfigurationListBuilder extends ConfigEntityListBuilder
{
  use MoLayoutBuilder;
  use MoGhostInvoker;

  /**
   * {@inheritdoc}
   */
  public function buildHeader(): array
  {
    $header['app_name'] = $this->t('App Name');
    $header['provider'] = $this->t('Provider');
    $header['client_id'] = $this->t('Client Id');
    $header['test_button'] = $this->t('Test');
    $header['mapping'] = $this->t('Mappings');
    $header['action'] = $this->t('Action');
    return $header;
  }

  public function buildOperations(EntityInterface $entity): array
  {
    $operations = parent::buildOperations($entity);

    $operations['#links']['edit']['url'] = Url::fromRoute('mo_configuration.edit', ['mo_client_config' => $entity->id()]);
    return $operations;
  }

  public function buildMappingOperations(EntityInterface $entity): array
  {
    return [
      '#type' => 'operations',
      '#links' => [
        'attribute_mapping' => [
          'title' => t('Attribute Mapping'),
          'weight' => 10,
          'url' => Url::fromRoute(
            'mo_configuration.edit', ['mo_client_config' => $entity->id(), 'tab' => 'attribute_mapping']
          )
        ],
        'role_mapping' => [
          'title' => t('Role Mapping'),
          'weight' => 20,
          'url' => Url::fromRoute(
            'mo_configuration.edit', ['mo_client_config' => $entity->id(), 'tab' => 'role_mapping']
          )
        ],
        'profile_mapping' => [
          'title' => t('Profile Mapping'),
          'weight' => 20,
          'url' => Url::fromRoute(
            'mo_configuration.edit', ['mo_client_config' => $entity->id(), 'tab' => 'profile_mapping']
          )
        ],
        'group_mapping' => [
          'title' => t('Group Mapping'),
          'weight' => 20,
          'url' => Url::fromRoute(
            'mo_configuration.edit', ['mo_client_config' => $entity->id(), 'tab' => 'group_mapping']
          )
        ],
        'client_setting' => [
          'title' => t('Client Settings'),
          'weight' => 20,
          'url' => Url::fromRoute(
            'mo_configuration.edit', ['mo_client_config' => $entity->id(), 'tab' => 'client_setting']
          )
        ]
      ]
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildRow(EntityInterface $entity): array
  {
    /** @var MoClientConfiguration $entity */
    $row['app_name']['data'] = Link::createFromRoute($entity->getAppName(), 'mo_configuration.edit', ['mo_client_config' => $entity->id()]);
    $row['provider']['data'] = $entity->getClientApp();
    $row['client_id'] = strlen($entity->getClientId()) > 25 ?
      substr($entity->getClientId(), 0, 24) . '...' :
      $entity->getClientId();
    $row['test_button']['data'] = self::getTestConfigMarkup($entity);
    $row['mapping']['data'] = $this->buildMappingOperations($entity);
    $row['action']['data'] = $this->buildOperations($entity);
    return $row;
  }

  /**
   * Get the test configuration button markup
   *
   * @param MoClientConfiguration $entity
   *  Application entity object
   *
   * @return array
   */
  public static function getTestConfigMarkup(MoClientConfiguration $entity): array
  {
    $button = [
      '#type' => 'button',
      '#value' => t('Perform Test Configuration'),
      '#attributes' => [
        'class' => ['button', 'button--primary', 'button--small', 'mo-test-config'],
        'url' => Url::fromRoute(
          $entity->getGrantType() == 'password' ? 'mo_oauth.password_grant.test_config' : 'mo_oauth.authorization_request',
          ['mo_client_config' => $entity->id(), 'test_sso' => true],
          ['absolute' => TRUE]
        )->toString(),
      ],
      '#attached' => [
        'library' => [
          // Drupal new window library
          'miniorange_oauth_client/test_config_window_js'
        ],
      ],
    ];
    if (\Drupal::service('miniorange_oauth_client.uno_module_subscription_library')->getModuleStatus()) {
      $button['#attributes'] += ['disabled' => true];
    }
    return $button;
  }

  /**
   * Page render operation this is the last stage before rendering the page
   *
   * @return array
   * @throws \Exception
   */
  public function render(): array
  {
    // I declared a empty param and passed it to the function,
    // we need to declare were to add the libraries in the called functions.
    $parent = parent::render();
    // General AJAX library
    $parent['libraries']['#attached']['library'] = [
      'core/drupal.dialog.ajax',
      'miniorange_oauth_client/notify_us_popup',
      'miniorange_oauth_client/common_js',
    ];

    $add_new = $this->buildTopBar();
    $result = self::buildMOLayout($add_new + $parent);

    MoSupport::buildSupportIcon($result);
    return $result;
  }

  /**
   * This function is to build the top nav bar either with the string add new button.
   *
   * @return array
   * @throws \Exception
   */
  public function buildTopBar(): array
  {
    $my_list = count($this->storage->loadMultiple());
    $add_new = [];

    $license_config = $this->call([MoUnoLicense::class, 'readMe']);
    $provided_count = $license_config?->getNoOfSp() ?? 1;

    \Drupal::service('miniorange_oauth_client.uno_module_subscription_library')->getModuleStatus($add_new);

    if ($my_list > 0) {
      $add_new['attr_map_warning'] = MoClientConfigurationHelper::getAttributeMappingMarkup();
    }
      $add_new['client_config_add_button'] = [
        '#type' => 'link',
        '#id' => 'client-config-add-button',
        '#title' => $this->t('+ Add New ( ') . $provided_count - $my_list . $this->t(' Remaining )'),
        '#url' => Url::fromRoute('mo_oauth.add_client_config'),
        '#attributes' => [
          'class' => ['button', 'button--primary', 'button--small', 'use-ajax'],
          'data-dialog-type' => 'modal',
        ],
      ];

    return $add_new;
  }
}
