<?php

declare(strict_types=1);

namespace Drupal\miniorange_oauth_client\Form;

use Drupal\Core\Entity\EntityForm;
use Drupal\Core\Form\FormStateInterface;
use Drupal\miniorange_oauth_client\Entity\MoAttributeMapping;
use Drupal\miniorange_oauth_client\Entity\MoClientConfiguration;
use Drupal\miniorange_oauth_client\MoConstant\MoModuleConstant;
use Drupal\miniorange_oauth_client\MoFeatures\MoUnoFeatures\MoUnoLicenseTierManager;
use Drupal\miniorange_oauth_client\MoFeatures\MoUnoFeatures\MoUnoTestConfigAttributeListBuilder;
use Drupal\miniorange_oauth_client\MoFeatures\MoUnoFeatures\MoUnoTutorial;
use Drupal\miniorange_oauth_client\MoLibrary\MoGhostInvoker;
use Drupal\miniorange_oauth_client\MoService\MoAjaxTables;
use Drupal\miniorange_oauth_client\MoService\MoLayoutBuilder;

/**
 * MiniOrange Attribute Mapping Entity form.
 */
final class MoAttributeMappingForm extends EntityForm {

  use MoLayoutBuilder;
  use MoGhostInvoker;

  public function buildForm(array $form, FormStateInterface $form_state) {
    $form = parent::buildForm($form, $form_state);
    $disabled = \Drupal::service('miniorange_oauth_client.uno_module_subscription_library')->getModuleStatus($form);

    /** @var MoAttributeMapping $entity */
    $entity = $this->getEntity();
    $current_lic_ver = $this->call([MoUnoLicenseTierManager::class, 'getLicenseVersion']);

    $config_entity = MoClientConfiguration::load($entity->id());
    $server_attrs = array_merge(['' => "-- Select --"], $config_entity->getServerTestAttributeKeys());
    $form['entity_key'] = [
      '#type' => 'hidden',
      '#value' => $entity->id(),
    ];

    $form['attr_map_is_enabled'] = [
      '#type' => 'hidden',
      '#value' => (int)!$disabled,
    ];
    $form['email_attribute'] = [
      '#title' => t('Email Attribute'),
      '#type' => 'select',
      '#tier' => MoModuleConstant::MO_FREE_VERSION_TIER,
      '#id' => 'text_field',
      '#options' => $server_attrs,
      '#required' => true,
      '#default_value' => $entity->getEmailAttribute(),
      '#states' => [
        'disabled' => [
          [':input[name = "attr_map_is_enabled"]' => ['value' => '0']],
        ],
      ],
    ];

    $form['username_attribute'] = array(
      '#type' => 'select',
      '#id' => 'text_field',
      '#tier' => MoModuleConstant::MO_FREE_VERSION_TIER,
      '#title' => t('Username Attribute'),
      '#options' => $server_attrs,
      '#default_value' => $entity->getUsernameAttribute(),
      '#states' => [
        'disabled' => [
          [':input[name = "attr_map_is_enabled"]' => ['value' => '0']],
        ],
      ],
    );

    // this part is to build custom mapping table structure
    $all_user_fields = \Drupal::service('entity_field.manager')->getFieldStorageDefinitions('user', 'user');
    $select = ['' => '-- Select --'];

    $all_user_fields = $select + array_combine(array_keys($all_user_fields), array_keys($all_user_fields));

    if(count($all_user_fields) > 1)
      $custom_user_fields = array_diff(
        $all_user_fields,
        [
          'user_picture' => 'user_picture',
          'uid' => 'uid',
          'created' => 'created',
          'changed' => 'changed',
          'access' => 'access',
          'login' => 'login',
          'init' => 'init',
          'roles' => 'roles',
          'mail' => 'mail',
          'pass' => 'pass'
        ]
      );
    else
      $custom_user_fields = $select;

    $custom_mapping_rows = $entity->getCustomAttributes();

    $server_names_array = [1 => $server_attrs, 2 => $custom_user_fields];
    $custom_mapping_fields = self::customMappingFields();
    $custom_mapping_header = self::customMappingHeader();
    $custom_unique_id_array = MoAjaxTables::getUniqueID($form_state->get('custom_attribute_mapping_table_uid'), $custom_mapping_rows);
    $form_state->set('custom_attribute_mapping_table_uid', $custom_unique_id_array);

    $form['custom_attr_map_frame'] = [
      '#type' => 'details',
      '#tier' => MoModuleConstant::MO_STANDARD_VERSION_TIER,
      '#title' => t('Custom Attribute Mapping'),
      '#open' => TRUE,
      '#states' => [
        'disabled' => [
          [':input[name = "attr_map_is_enabled"]' => ['value' => '0']],
        ],
      ],
      'definition' => [
        "#type" => "markup",
        "#markup" => t(
            "Select the Drupal field name in the <i> Drupal Attribute Machine Name </i> dropdown and add the
           OAuth Server attributes that you need to map with the Drupal machine name in the
           <i>OAuth Server Attribute Name dropdown</i><br><br>"
        )
      ],
      'mapping' => [
        'custom_attributes' => MoAjaxTables::generateTables(
          'custom-attribute-mapping-table',
          $custom_mapping_fields,
          $custom_unique_id_array,
          $custom_mapping_rows,
          $custom_mapping_header,
          $server_names_array
        ),
        'add_row' => MoAjaxTables::generateAddButton(
          'Add',
          [$this, 'addRowNew'],
          [$this, 'ajaxCallback'],
          'custom-attribute-mapping-table',
          '',
          $disabled || !in_array($current_lic_ver, MoModuleConstant::MO_STANDARD_VERSION_TIER)
        )
      ]
    ];
    // add additional properties to the base form submit operation
    if (!$entity->isNew()) {
      $form['actions']['delete'] = [
        '#type' => 'submit',
        '#value' => $this->t('Reset'),
        "#weight" => 2,
        '#disabled' => $disabled,
        '#attributes' => ['class' => ['button', 'button--danger']],
        '#submit' => ['::delete']
      ];
    }
    $form['actions']['submit']['#attributes']['class'] = ['button', 'button--primary'];
    $form['actions']['submit']['#weight'] = 1;
    $form['actions']['submit']['#disabled'] = $disabled;


    $result['mo_outer_container'] = [
      "#type" => "container",
      "#attributes" => ['class' => ['mo-outer-container']]
    ];

    $this->modifyFormTitleOnTier($form);
    $result['mo_outer_container'] += $this->buildMOLayout($form);

    $result['mo_outer_container']['tutorial_frame'] = self::getTutorialLayout();

    $this->call(
      [MoUnoTutorial::class, 'buildVideoTutorialFrame'],
      ['result' => &$result['mo_outer_container']['tutorial_frame']]
    );

    $this->call(
      [MoUnoTestConfigAttributeListBuilder::class, 'buildTestConfigurationFrame'],
      ['result' => &$result['mo_outer_container']['tutorial_frame'], 'entity_id' => $entity->id()]
    );


    return $result;
  }
  public function submitForm(&$form, FormStateInterface $form_state): void
  {
    $form_state->setRedirect(
      'mo_configuration.edit', ['mo_client_config' => $this->entity->id(), 'tab' => 'attribute_mapping']
    );
    parent::submitForm($form, $form_state);
    \Drupal::messenger()->addMessage("Mapped attributes are saved successfully.");
  }
  public function delete(array $form, FormStateInterface $form_state): void
  {
    try {
      $this->entity->delete();
      \Drupal::messenger()->addMessage("Mapped attributes are removed successfully.");
    } catch (\Exception $exception) {
      //todo need to add catch block
    }
  }
  public function customMappingFields()
  {
    return [
      'oauth_attribute_name' => [
        'type' => 'select',
      ],
      'drupal_attribute_name' => [
        'type' => 'select',
      ],
      'delete_button_role' => [
        'type' => 'submit',
        'submit' => [$this, 'removeRow'],
        'callback' => [$this, 'ajaxCallback'],
        'wrapper' => 'custom-attribute-mapping-table',
      ],
    ];
  }
  public function customMappingHeader()
  {
    return [
      [
        'data' => t('OAuth Server Attribute'),
        'width' => '45%',
      ],
      [
        'data' => t('Drupal Machine Attribute'),
        'width' => '45%',
      ],
      [
        'data' => t('Action'),
        'width' => '10%',
      ],
    ];
  }
  public function addRowNew(array &$form, FormStateInterface $form_state)
  {
    $add_button = ($form_state->getTriggeringElement()['#name']);
    $rows = $form_state->getValue('total_rows_' . $add_button);
    $unique_array_id = $form_state->get('custom_attribute_mapping_table_uid');
    for ($count = 1; $count <= $rows; $count++) {
      $uuid_service = \Drupal::service('uuid');
      $uuid = $uuid_service->generate();
      $unique_array_id[] = $uuid;
    }
    $form_state->set('custom_attribute_mapping_table_uid', $unique_array_id);
    $form_state->setRebuild();
  }
  public function ajaxCallback(array &$form, FormStateInterface $form_state)
  {
    return $form['mo_outer_container']['mo_outline']['custom_attr_map_frame']['mapping']['custom_attributes'];
  }
  public function removeRow(array &$form, FormStateInterface $form_state)
  {
    $id = $form_state->getTriggeringElement()['#name'];
    $var_value = 'custom_attribute_mapping_table_uid';
    $unique_id = $form_state->get($var_value);
    $unique_id = array_diff($unique_id, [$id]);
    if (empty($unique_id)) {
      $uuid_service = \Drupal::service('uuid');
      $unique_id[0] = $uuid_service->generate();
    }
    $form_state->set($var_value, $unique_id);
    $form_state->setRebuild();
  }
}
