<?php

namespace Drupal\miniorange_oauth_client\Form;

use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\InsertCommand;
use Drupal\Core\Ajax\InvokeCommand;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\TempStore\PrivateTempStoreFactory;
use Drupal\Core\Url;
use Drupal\miniorange_oauth_client\Entity\MoAttributeMapping;
use Drupal\miniorange_oauth_client\Entity\MoClientConfiguration;
use Drupal\miniorange_oauth_client\Entity\MoClientSettings;
use Drupal\miniorange_oauth_client\Entity\MoProfileMapping;
use Drupal\miniorange_oauth_client\Entity\MoRoleMapping;
use Drupal\miniorange_oauth_client\MoConstant\MoModuleConstant;
use Drupal\miniorange_oauth_client\MoDTO\MoModuleSettings;
use Drupal\miniorange_oauth_client\MoFeatures\MoUnoFeatures\MoUnoLicenseTierManager;
use Drupal\miniorange_oauth_client\MoHelper\MoSupport;
use Drupal\miniorange_oauth_client\MoLibrary\MoGhostInvoker;
use Drupal\miniorange_oauth_client\MoLibrary\MoLogger;
use Drupal\miniorange_oauth_client\MoService\MoLayoutBuilder;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\File\UploadedFile;

class MoConfigurationImportExport extends formBase
{
  use MoLayoutBuilder;
  use MoGhostInvoker;

  protected $tempStore;

  public function __construct(PrivateTempStoreFactory $temp_store_factory)
  {
    $this->tempStore = $temp_store_factory->get(MoModuleConstant::MODULE_NAME);
  }

  public static function create(ContainerInterface $container)
  {
    return new static(
      $container->get('tempstore.private')
    );
  }

  public function getFormId()
  {
    return "moConfigurationImportExport";
  }

  public function buildForm(array $form, FormStateInterface $form_state)
  {
    $form['markup_library'] = [
      '#attached' => [
        'library' => [
          // ajax js library
          "core/drupal.dialog.ajax",
          // common js library to remove the drupal message
          'miniorange_oauth_client/common_js',
        ]
      ],
    ];

    $form['message_wrapper'] = [
      '#type' => 'container',
      '#attributes' => ['id' => 'ajax-messages'],
    ];

    $disabled = \Drupal::service('miniorange_oauth_client.uno_module_subscription_library')->getModuleStatus($form);

    $this->buildExportFrame($form);
    $this->buildImportFrame($form, $disabled);

    $this->modifyFormTitleOnTier($form);
    $result = $this->buildMOLayout($form);
    MoSupport::buildSupportIcon($result);
    return $result;
  }


  public function buildExportFrame(array &$form): void
  {
    $form['export_frame'] = [
      '#type' => 'fieldset',
      '#tier' => MoModuleConstant::MO_FREE_VERSION_TIER,
      '#title' => t('Export Configuration'),
      'info' => [
        '#type' => 'item',
        '#description' => t('Download your module configuration to use on another Drupal instance.').'<br>'.
          t('Click').' <b>'.t('Download').'</b> '. t('to export the configuration as a JSON file.')
      ]
    ];

    $is_configured = FALSE;
    if (empty(MoClientConfiguration::loadMultiple())) {
      $form['export_frame']['oauth_imo_option_export_msg'] = array(
        '#markup' => '<div class="mo-register-message">Please <a href="' . Url::fromRoute('entity.mo_client_config.add_form')->toString() . '">configure the module</a> first to export the configurations.</div><br>',
      );
      $is_configured = TRUE;
    }


    $form['export_frame']['conf_export'] = array(
      '#type' => 'submit',
      '#tier' => MoModuleConstant::MO_FREE_VERSION_TIER,
      '#value' => t('Download'),
      '#button_type' => 'primary',
      '#ajax' => [
        'callback' => '::miniorange_export',
        'progress' => ['type' => 'throbber'],
      ],
      '#disabled' => $is_configured,
    );
  }

  public function buildImportFrame(array &$form, bool $disabled): void
  {
    $form['import_frame'] = [
      '#type' => 'fieldset',
      '#tier' => MoModuleConstant::MO_FREE_VERSION_TIER,
      '#title' => t('Import Configuration'),
      'info' => [
        'line_1' => [
          '#type' => 'item',
          '#description' => t(
            'Import module configurations from another Drupal instance.').' <br>'.
            t('Upload the JSON file previously exported from this module, then click ') .'<b>'.t('Validate & Upload.').'</b>'
        ],
      ],
    ];

    $form['import_frame']['config_file'] = array(
      '#type' => 'file',
      '#tier' => MoModuleConstant::MO_FREE_VERSION_TIER,
      '#disabled' => $disabled,
    );

    $form['import_frame']['import_button'] = array(
      '#type' => 'submit',
      '#tier' => MoModuleConstant::MO_FREE_VERSION_TIER,
      '#value' => t('Validate & Upload'),
      '#button_type' => 'primary',
      '#submit' => array('::miniorange_import_Config_helper'),
      '#disabled' => $disabled,
    );
  }

  public function submitForm(array &$form, FormStateInterface $form_state)
  {
    // TODO: Implement submitForm() method.
  }

  public function miniorange_export()
  {
    $client_configs = MoClientConfiguration::loadMultiple();
    foreach ($client_configs as $config) {
      $config_array = $config->toArray();
      unset($config_array['server_test_attributes']);
      $arr[$config->id()]['Client_config'] = $config_array;
      $arr[$config->id()]['role_map'] = MoRoleMapping::load($config->id())?->toArray();
      $arr[$config->id()]['attr_map'] = MoAttributeMapping::load($config->id())?->toArray();
      $arr[$config->id()]['profile_map'] = MoProfileMapping::load($config->id())?->toArray();
      $arr[$config->id()]['client_setting'] = MoClientSettings::load($config->id())?->toArray();
    }
    $arr['module_settings'] = MoModuleSettings::readMe()->toArray();

    $data = json_encode($arr, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);

    $this->tempStore->set('download_config_data', $data);

    $filename = 'miniorange_oauth_config' . date('Y-m-d_H-i-s') . '.json';

    $this->messenger()->addStatus($this->t('Configuration is ready, Download will start shortly'));

    $response = new AjaxResponse();

    $message = ['#type' => 'status_messages'];

    $response->addCommand(new InsertCommand('#ajax-messages', \Drupal::service('renderer')->renderRoot($message)));

    $download_url = URL::fromRoute('mo_oauth.download_config', [], ['absolute' => true])->toString();

    $iframe_html = "<iframe style='display:none' src='$download_url'></iframe>";
    $response->addCommand(new InvokeCommand('body', 'append', [$iframe_html]));

    return $response;

  }
  public function miniorange_import_Config_helper(array &$form, FormStateInterface $form_state): void
  {
    try {
      $request = \Drupal::requestStack()->getCurrentRequest();

      /** @var UploadedFile|null $uploaded_file */
      $uploaded_file = $request->files->get('files')['config_file'] ?? null;

      if (!$uploaded_file || !$uploaded_file->isValid()) {
        \Drupal::messenger()->addError($this->t('Please upload a valid file.'));
        return;
      }

      // Validate extension manually
      $extension = strtolower($uploaded_file->getClientOriginalExtension());
      if ($extension !== 'json') {
        \Drupal::messenger()->addError($this->t('Only JSON files are allowed.'));
        return;
      }

      // Read and decode JSON directly without saving
      $content = file_get_contents($uploaded_file->getRealPath());
      $configurations = json_decode($content, TRUE);

      if (json_last_error() !== JSON_ERROR_NONE) {
        \Drupal::messenger()->addError($this->t('The uploaded file contains invalid JSON.'));
        return;
      }

      if (!is_array($configurations)) {
        \Drupal::messenger()->addError($this->t('Invalid or empty JSON structure.'));
        return;
      }

      $entity_classes = [
        'Client_config' => MoClientConfiguration::class,
        'role_map' => MoRoleMapping::class,
        'attr_map' => MoAttributeMapping::class,
        'profile_map' => MoProfileMapping::class,
        'client_setting' => MoClientSettings::class,
      ];

      $updated_ids = [];
      $created_ids = [];

      foreach ($configurations as $id => $items) {
        foreach ($entity_classes as $key => $class) {
          if (!empty($items[$key]) && is_array($items[$key])) {
            $data = $items[$key];
            unset($data['uuid']);

            $entity = $class::load($id);
            if ($entity) {
              foreach ($data as $field => $value) {
                $entity->set($field, $value);
              }
              $updated_ids[$class][] = $id;
            } else {
              $data['id'] = $id;
              $entity = $class::create($data);
              $created_ids[$class][] = $id;
            }
            $entity->save();
          }
        }
      }
      if (!empty($updated_ids[MoClientConfiguration::class])) {
        foreach ($updated_ids[MoClientConfiguration::class] as $updated_id) {
          $client_config = MoClientConfiguration::load($updated_id);
          $name = $client_config?->get('custom_name') ?? $updated_id;
          \Drupal::messenger()->addStatus($this->t('Configuration for "@name" already exist! and has been updated.', ['@name' => $name]));
        }
      }
      if (!empty($created_ids[MoClientConfiguration::class])) {
        foreach ($created_ids[MoClientConfiguration::class] as $created_id) {
          $client_config = MoClientConfiguration::load($created_id);
          $name = $client_config?->get('custom_name') ?? $created_id;
          \Drupal::messenger()->addStatus($this->t('Configuration for "@name" has been imported successfully.', ['@name' => $name]));
        }
      }
      if (empty($updated_ids) && empty($created_ids)) {
        \Drupal::messenger()->addStatus($this->t('Configuration imported successfully.'));
      }

      //DO-451
      $form_state->setRedirect('entity.mo_client_config.collection');
      \Drupal::messenger()->addStatus("Please click on the 'Perform Test Configuration' button to proceed with mapping.");

    } catch (\Exception $e) {
      MoLogger::error($e->getMessage());
      \Drupal::messenger()->addError($e->getMessage());
    }
  }
}
