<?php

declare(strict_types=1);

namespace Drupal\miniorange_oauth_client\Form;

use Drupal\Core\Entity\EntityForm;
use Drupal\Core\Form\FormStateInterface;
use Drupal\miniorange_oauth_client\MoFeatures\MoUnoFeatures\MoLicense\MoLibrary\MoUnoSubscriptionLibrary;
use Drupal\miniorange_oauth_client\MoFeatures\MoUnoFeatures\MoUnoLicenseTierManager;
use Drupal\miniorange_oauth_client\MoConstant\MoModuleConstant;
use Drupal\miniorange_oauth_client\MoFeatures\MoUnoFeatures\MoUnoTestConfigAttributeListBuilder;
use Drupal\miniorange_oauth_client\MoLibrary\MoGhostInvoker;
use Drupal\miniorange_oauth_client\MoService\MoLayoutBuilder;
use Drupal\miniorange_oauth_client\MoFeatures\MoUnoFeatures\MoUnoTutorial;
use Drupal\Core\Link;
use Drupal\Core\Url;

use Drupal\miniorange_oauth_client\MoService\MoAjaxTables;

/**
 * Miniorange Group Mapping Entity form.
 */
final class MoGroupMappingForm extends EntityForm {

  use MoLayoutBuilder;
  use MoGhostInvoker;
  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array
  {
    /** @var MoUnoSubscriptionLibrary $sub_lib */
    $sub_lib = \Drupal::service('miniorange_oauth_client.uno_module_subscription_library');
    $disabled = $sub_lib->getModuleStatus($form);

    $group_module_enabled = \Drupal::moduleHandler()->moduleExists('group');
    if (!$group_module_enabled) {
      $group_module_url = Url::fromUri('https://www.drupal.org/project/group', ['attributes' => ['target' => '_blank']]);
      $group_link = Link::fromTextAndUrl('Group module', $group_module_url)->toString();

      $form['group_mapping_not_available'] = [
        '#type' => 'markup',
        '#prefix' => '<div class="messages messages--warning">',
        '#suffix' => '</div>',
        '#markup' => $this->t('This extends the functionality of @group_module', ['@group_module' => $group_link]),
      ];
    }

    $form = parent::buildForm($form, $form_state);
    $entity = $this->getEntity();
    $current_lic_ver = $this->call([MoUnoLicenseTierManager::class, 'getLicenseVersion']);

    $form['entity_key'] = [
      '#type' => 'hidden',
      '#value' => $entity->id(),
    ];

    $form['group_map_is_enabled'] = [
      '#type' => 'hidden',
      '#value' => (int)(!$disabled && $group_module_enabled),
    ];

    $form['enable_group_mapping'] = [
      '#type' => 'checkbox',
      '#tier' => MoModuleConstant::MO_ENTERPRISE_VERSION_TIER,
      '#title' => t('Enable Group Mapping'),
      '#default_value' => $entity->getEnableGroupMapping() ,
      '#description' => t('Enable this first if you want to use Group Mapping.'),
      '#states' => [
        'disabled' => [
          [':input[name="group_map_is_enabled"]' => ['value' => '0']],
        ],
      ]
    ];

    $form['keep_existing_groups'] = [
      '#type' => 'checkbox',
      '#tier' => MoModuleConstant::MO_ENTERPRISE_VERSION_TIER,
      '#title' => t('Keep existing groups if groups not mapped below'),
      '#default_value' => $entity->getKeepExistingGroups() ,
      '#states' => [
        'disabled' => [
          [':input[name = "enable_group_mapping"]' => ['checked' => FALSE]],
        ]
      ]
    ];

    $form['group_attribute'] = [
      '#type' => 'textfield',
      '#tier' => MoModuleConstant::MO_ENTERPRISE_VERSION_TIER,
      '#title' => t('Group Attribute'),
      '#default_value' => $entity->getGroupAttribute(),
      '#attributes' => ['placeholder' => 'Enter Group Attribute'],
      '#states' => [
        'disabled' => [
          [':input[name = "enable_group_mapping"]' => ['checked' => FALSE]],
        ]
      ],
      '#required' => FALSE,
    ];

    $group_mapping = $entity->getMappedFields();
    $group_mapping_fields = $this->GroupMappingFields($group_module_enabled);
    $group_unique_id_array = MoAjaxTables::getUniqueID($form_state->get('group_mapping_table_uid'), $group_mapping);

    if (empty($group_unique_id_array)) {
      $uuid_service = \Drupal::service('uuid');
      $group_unique_id_array[] = $uuid_service->generate();
    }

    $form_state->set('group_mapping_table_uid', $group_unique_id_array);
    $server_attrs = $this->getGroupFields($group_module_enabled);
    $group_mapping_header = $this->GroupMappingHeader();
    $server_names_array = [1 => $server_attrs, 2 => $group_mapping];


    // Implementing the original group mapping form.
    $groupsCreated = [];
    $groups = $group_module_enabled ? \Drupal::entityTypeManager()->getStorage('group')->loadMultiple() : [];
    $groupsDisplayed = [];
    foreach ($groups as $group) {
      $groupsCreated[$group->id()] = trim($group->label());
      $groupsDisplayed[trim($group->label())] = trim($group->label());
    }
    $form['group_mapping_settings_frame'] = [
      '#type' => 'details',
      '#tier' => MoModuleConstant::MO_ENTERPRISE_VERSION_TIER,
      '#title' => t('Group Mapping Settings'),
      '#open' => TRUE,
      '#states' => [
        'disabled' => [
          [':input[name = "enable_group_mapping"]' => ['checked' => FALSE]],
        ],
      ],
      'mapping' => [
        'mapped_fields' => MoAjaxTables::generateTables(
          'group-mapping-table',
          $group_mapping_fields,
          $group_unique_id_array,
          $group_mapping,
          $group_mapping_header,
          $server_names_array
        ),
        'add_row' => MoAjaxTables::generateAddButton(
          'Add',
          [$this, 'addRowNew'],
          [$this, 'ajaxCallback'],
          'group-mapping-table',
          '',
          $disabled || !in_array($current_lic_ver, MoModuleConstant::MO_ENTERPRISE_VERSION_TIER)
        )
      ],
    ];


      $form['actions']['delete'] = [
        '#type' => 'submit',
        '#value' => $this->t('Reset'),
        "#weight" => 2,
        '#states' => [
          'disabled' => [
            [':input[name="enable_group_mapping"]' => ['checked' => FALSE]],
          ],
        ],
        '#attributes' => ['class' => ['button', 'button--danger']],
        '#submit' => ['::delete']
      ];

    $form['actions']['submit']['#attributes']['class'] = ['button', 'button--primary'];
    $form['actions']['submit']['#weight'] = 1;
    $form['actions']['submit']['#states'] = [
      'disabled' => [
        [':input[name="enable_group_mapping"]' => ['checked' => FALSE]],
      ],
    ];


    $result['mo_outer_container'] = [
      "#type" => "container",
      "#attributes" => ['class' => ['mo-outer-container']]
    ];
    $this->modifyFormTitleOnTier($form);
    $result['mo_outer_container'] += $this->buildMOLayout($form);

    $result['mo_outer_container']['tutorial_frame'] = self::getTutorialLayout();

    $this->call(
      [MoUnoTutorial::class, 'buildVideoTutorialFrame'],
      ['result' => &$result['mo_outer_container']['tutorial_frame']]
    );

    $this->call(
      [MoUnoTestConfigAttributeListBuilder::class, 'buildTestConfigurationFrame'],
      ['result' => &$result['mo_outer_container']['tutorial_frame'], 'entity_id' => $entity->id()]
    );

    return $result;

  }

  public function submitForm(&$form, FormStateInterface $form_state): void
  {
    $form_state->setRedirect(
      'mo_configuration.edit', ['mo_client_config' => $this->entity->id(), 'tab' => 'group_mapping']
    );
    parent::submitForm($form, $form_state);
  }

  public function getGroupFields($group_module_enabled): array {
    $options = [];
    $groups = $group_module_enabled ? \Drupal::entityTypeManager()
      ->getStorage('group')
      ->loadMultiple() : [];

    foreach ($groups as $group) {
      /** @var \Drupal\group\Entity\Group $group */
      $options[$group->id()] = $group->label();
    }

    return $options;
  }

  public function addRowNew(array &$form, FormStateInterface $form_state)
  {
    $add_button = ($form_state->getTriggeringElement()['#name']);
    $rows = $form_state->getValue('total_rows_' . $add_button);
    $unique_array_id = $form_state->get('group_mapping_table_uid');
    for ($count = 1; $count <= $rows; $count++) {
      $uuid_service = \Drupal::service('uuid');
      $uuid = $uuid_service->generate();
      $unique_array_id[] = $uuid;
    }
    $form_state->set('group_mapping_table_uid', $unique_array_id);
    $form_state->setRebuild();
  }
  public function ajaxCallback(array &$form, FormStateInterface $form_state) {
    return $form['mo_outer_container']['mo_outline']['group_mapping_settings_frame']['mapping']['mapped_fields'];
  }

  public function removeRow(array &$form, FormStateInterface $form_state)
  {
    $id = $form_state->getTriggeringElement()['#name'];
    $var_value = 'group_mapping_table_uid';
    $unique_id = $form_state->get($var_value);

    $mapped_fields = $this->getEntity()->getMappedFields() ?? [];
    $unique_id = array_diff($unique_id, [$id]);
    if (empty($unique_id)) {
      $uuid_service = \Drupal::service('uuid');
      $unique_id[0] = $uuid_service->generate();
    }
    if (isset($mapped_fields[$id])) {
      unset($mapped_fields[$id]);
      $this->getEntity()->setMappedFields($mapped_fields);
    }

    $form_state->set($var_value, $unique_id);
    $form_state->setRebuild();
  }

  public function delete(array $form, FormStateInterface $form_state): void
  {
    try {
      $this->entity->delete();
    } catch (\Exception $exception) {
      //todo need to add catch block
    }
  }

  public function GroupMappingHeader(){
    return [
      ['data' => t('Drupal Group Name')],
      ['data' => t('OAuth Server Group Name/ID')],
      ['data' => t('Action')],
    ];
  }

  public function GroupMappingFields($group_module_enabled): array {
    // Get available groups for the select options
    $groups = $group_module_enabled ? \Drupal::entityTypeManager()->getStorage('group')->loadMultiple() : [];
    $group_options = ['' => t('-- Select --')];
    foreach ($groups as $group) {
      $group_options[$group->id()] = $group->label();
    }

    return [
      'drupal_group_name' => [
        'type' => 'select',
        'options' => $group_options,
        'attributes' => ['class' => ['drupal-group-select']],
      ],
      'oauth_group_name' => [
        'type' => 'textfield',
        'placeholder' => 'Semicolon (;) seperated values',
      ],
      'delete_button_role' => [
        'type' => 'submit',
        'submit' => [$this, 'removeRow'],
        'callback' => [$this, 'ajaxCallback'],
        'wrapper' => 'group-mapping-table',
      ],
    ];
  }

}

