<?php

declare(strict_types=1);

namespace Drupal\miniorange_oauth_client\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Link;
use Drupal\Core\Render\Markup;
use Drupal\Core\Url;
use Drupal\miniorange_oauth_client\Entity\MoClientConfiguration;
use Drupal\miniorange_oauth_client\MoConstant\MoModuleConstant;
use Drupal\miniorange_oauth_client\MoDTO\MoModuleSettings;
use Drupal\miniorange_oauth_client\MoFeatures\MoUnoFeatures\MoUnoLicenseTierManager;
use Drupal\miniorange_oauth_client\MoFeatures\MoUnoFeatures\MoUnoTutorial;
use Drupal\miniorange_oauth_client\MoHelper\MoSupport;
use Drupal\miniorange_oauth_client\MoLibrary\MoGhostInvoker;
use Drupal\miniorange_oauth_client\MoService\MoAjaxTables;
use Drupal\miniorange_oauth_client\MoService\MoLayoutBuilder;

/**
 * Configure MiniOrange OAuth Client settings for this site.
 */
final class MoModuleSettingsForm extends ConfigFormBase
{
  use MoLayoutBuilder;
  use MoGhostInvoker;

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string
  {
    return 'mo_module_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array
  {
    return ['miniorange_oauth_client.module_settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array
  {
    $form = parent::buildForm($form, $form_state);

    $entity = MoModuleSettings::readMe();

    $disabled = \Drupal::service('miniorange_oauth_client.uno_module_subscription_library')->getModuleStatus($form);

    $current_lic_ver = $this->call([MoUnoLicenseTierManager::class, 'getLicenseVersion']);

    $form['libraries'] = [
      '#attached' => [
        'library' => [
          // CSS library for showing the client configuration error message
          'miniorange_oauth_client/common_layout',
          // CSS library that handles the highlighted text basically for
          // 1. Back door login
          'miniorange_oauth_client/module_settings_page_layout',
        ]
      ]
    ];
    $form['mod_sett_is_enabled'] = [
      '#type' => 'hidden',
      '#value' => (int)!$disabled,
    ];

    $form['enforce_https_callback'] = [
      '#type' => 'checkbox',
      '#tier' => MoModuleConstant::MO_FREE_VERSION_TIER,
      '#title' => t('Enforce ') . '<b>' . t('HTTPS') . '</b>' . t(' Callback URL'),
      '#description' => t('Check this option if the OAuth Provider only support HTTPS Callback URL and you have an HTTP site.'),
      '#default_value' => $entity->getEnforceHttpsCallback(),
      '#states' => [
        'disabled' => [
          [':input[name="mod_sett_is_enabled"]' => ['value' => '0']],
        ],
      ]
    ];
    $this->buildSiteProtection($form, $form_state, $disabled, $entity, $current_lic_ver);
    $this->buildDomainRestriction($form, $form_state, $disabled, $entity, $current_lic_ver);
    $this->buildUserActions($form, $form_state, $disabled, $entity, $current_lic_ver);

    $form['token_storage_frame'] = [
      '#type' => 'details',
      '#tier' => MoModuleConstant::MO_ENTERPRISE_VERSION_TIER,
      '#title' => t('Token Storage'),
      '#open' => TRUE,
      '#states' => [
        'disabled' => [
          [':input[name="mod_sett_is_enabled"]' => ['value' => '0']],
        ],
      ]
    ];

    $form['token_storage_frame']['store_token_in_session'] = [
      '#type' => 'checkbox',
      '#tier' => MoModuleConstant::MO_ENTERPRISE_VERSION_TIER,
      '#title' => t('Store Token in User Session'),
      '#default_value' => $entity->getStoreTokenInSession(),
      '#states' => [
        'disabled' => [
          [':input[name="mod_sett_is_enabled"]' => ['value' => '0']],
        ],
      ]
    ];
    $form['token_storage_frame']['store_token_in_cookies'] = [
      '#type' => 'checkbox',
      '#tier' => MoModuleConstant::MO_ENTERPRISE_VERSION_TIER,
      '#title' => t('Store Token in Browser Cookies'),
      '#default_value' => $entity->getStoreTokenInCookies(),
      '#states' => [
        'disabled' => [
          [':input[name="mod_sett_is_enabled"]' => ['value' => '0']],
        ],
      ]
    ];
    $form['token_storage_frame']['token_storage_keys_wrapper'] = [
      '#type' => 'container',
      '#states' => [
        'visible' => [
          [':input[name="store_token_in_cookies"]' => ['checked' => TRUE]],
          'or',
          [':input[name="store_token_in_session"]' => ['checked' => TRUE]],
        ],
        'disabled' => [
          [':input[name="mod_sett_is_enabled"]' => ['value' => '0']],
        ],
      ],
      'token_storage_keys' => [
        '#type' => 'table',
        '#responsive' => TRUE,
        '#header' => $this->tokenStorageTableHeader(),
        '#states' => [
          'disabled' => [
            [':input[name="mod_sett_is_enabled"]' => ['value' => '0']],
          ],
        ]
      ]
    ];
    $this->generateTokenStorageRows($entity->getTokenStorageKeys(), $form['token_storage_frame']['token_storage_keys_wrapper']['token_storage_keys']);


    $form['actions']['delete'] = [
      '#type' => 'submit',
      '#value' => $this->t('Reset'),
      '#submit' => [[MoModuleSettings::class, 'remove']],
      '#attributes' => [
        'class' => ['button', 'button--danger'],
      ],
      '#states' => [
        'disabled' => [
          [':input[name="mod_sett_is_enabled"]' => ['value' => '0']],
        ],
      ]
    ];
    $form['actions']['submit'] += [
      '#states' => [
        'disabled' => [
          [':input[name="mod_sett_is_enabled"]' => ['value' => '0']],
        ],
      ]
    ];

    $result['mo_outer_container'] = [
      "#type" => "container",
      "#attributes" => ['class' => ['mo-outer-container']]
    ];
    $this->modifyFormTitleOnTier($form);
    $result['mo_outer_container'] += $this->buildMOLayout($form);

    $result['mo_outer_container']['tutorial_frame'] = self::getTutorialLayout();

    $this->call(
      [MoUnoTutorial::class, 'buildVideoTutorialFrame'],
      ['result' => &$result['mo_outer_container']['tutorial_frame']]
    );
    MoSupport::buildSupportIcon($result);
    return $result;
  }

  private function generateTokenStorageRows(array $rows, array &$form): void
  {
    $fields = $this->tokenStorageTableFields();
    foreach ($rows as $key => $row) {
      $new_row = $fields;
      foreach ($row as $field => $value) {
        $new_row[$field]['#default_value'] = $value;
        if ($field == 'token_type')
          $new_row[$field]['#markup'] = $value;
      }
      $form[$key] = $new_row;
    }
  }

  private function buildUserActions(array &$form, FormStateInterface $form_state, bool $disabled, MoModuleSettings $entity): void
  {
    $form['user_actions_frame'] = [
      '#type' => 'details',
      '#tier' => MoModuleConstant::MO_ENTERPRISE_VERSION_TIER,
      '#title' => t('User Actions'),
      '#open' => TRUE,
      '#states' => [
        'disabled' => [
          [':input[name="mod_sett_is_enabled"]' => ['value' => '0']],
        ],
      ]
    ];

    $form['user_actions_frame']['make_profile_fields_required'] = [
      '#type' => 'checkbox',
      '#tier' => MoModuleConstant::MO_ENTERPRISE_VERSION_TIER,
      '#title' => t('Make selected user profile attributes required'),
      '#default_value' => $entity->getMakeProfileFieldsRequired(),
      '#states' => [
        'disabled' => [
          [':input[name="mod_sett_is_enabled"]' => ['value' => '0']],
        ],
      ]
    ];

    $form['user_actions_frame']['user_profile_required_attributes'] = [
      '#type' => 'textarea',
      '#disabled' => $disabled,
      '#tier' => MoModuleConstant::MO_ENTERPRISE_VERSION_TIER,
      '#default_value' => $entity->getUserProfileRequiredAttributes(),
      '#description' => 'As soon as user get created after first SSO, he will be redirected to the user profile page to enter mandatory fields.',
      '#attributes' => array('placeholder' => 'Enter semicolon(;) separated profile attribute machine names that you want to make mandatory.'),
      '#states' => [
        'disabled' => [
          [':input[name="mod_sett_is_enabled"]' => ['value' => '0']],
        ],
        'visible' => [
          ':input[name="make_profile_fields_required"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['user_actions_frame']['make_profile_field_disable'] = [
      '#type' => 'checkbox',
      '#tier' => MoModuleConstant::MO_ENTERPRISE_VERSION_TIER,
      '#title' => t('Disable users profile attribute fields'),
      '#default_value' => $entity->getMakeProfileFieldDisable(),
      '#states' => [
        'disabled' => [
          [':input[name="mod_sett_is_enabled"]' => ['value' => '0']],
        ],
      ]
    ];

    $form['user_actions_frame']['user_profile_disable_attributes_frame'] = [
      '#type' => 'fieldset',
      '#states' => [
        'disabled' => [
          [':input[name="mod_sett_is_enabled"]' => ['value' => '0']],
        ],
        'visible' => [
          ':input[name="make_profile_field_disable"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['user_actions_frame']['user_profile_disable_attributes_frame']['user_profile_disable_attributes'] = [
      '#type' => 'textarea',
      '#tier' => MoModuleConstant::MO_ENTERPRISE_VERSION_TIER,
      '#title' => t('Enter semicolon(;) separated profile attribute machine names that needs to be disabled'),
      '#default_value' => $entity->getUserProfileDisableAttributes(),
      '#description' => 'The users would not be able to changes these attributes.',
      '#attributes' => [
        'placeholder' => 'Enter semicolon(;) separated profile attribute machine names that you disable.'
      ],
      '#states' => [
        'disabled' => [
          [':input[name="mod_sett_is_enabled"]' => ['value' => '0']],
        ],
      ]
    ];

    $form['user_actions_frame']['user_profile_disable_attributes_frame']['password_and_confirm_password_status'] = [
      '#title' => t('Disable/Hide "<u>Password</u>" and "<u>Current Password</u>" fields of user profile page'),
      '#tier' => MoModuleConstant::MO_ENTERPRISE_VERSION_TIER,
      '#type' => 'radios',
      '#options' => [
        'editable' => t('Keep Editable'),
        'disable' => t('Disable'),
        'hide' => t('Disable and Hide'),
      ],
      '#default_value' => $entity->getPasswordAndConfirmPasswordStatus(),
      '#attributes' => ['class' => ['container-inline']],
      '#required' => TRUE,
      '#states' => [
        'disabled' => [
          [':input[name="mod_sett_is_enabled"]' => ['value' => '0']],
        ],
      ]
    ];

    $form['user_actions_frame']['duplicate_username'] = [
      '#type' => 'checkbox',
      '#tier' => MoModuleConstant::MO_ENTERPRISE_VERSION_TIER,
      '#title' => t('Allow duplicate username for user creation'),
      '#default_value' => $entity->getDuplicateUsername(),
      '#description' => t('During user creation in Drupal, if a user with the same username already exists, the feature will generate an incremented username. For example, if username \'xyz\' exists, the new users will be created as \'xyz01,\' \'xyz02,\' and so on.'),
      '#states' => [
        'disabled' => [
          [':input[name="mod_sett_is_enabled"]' => ['value' => '0']],
        ],
      ]
    ];

    $form['user_actions_frame']['change_status_to_active_on_sso'] = [
      '#type' => 'checkbox',
      '#tier' => MoModuleConstant::MO_ENTERPRISE_VERSION_TIER,
      '#default_value' => $entity->getChangeStatusToActiveOnSso(),
      '#title' => t('Activate blocked users upon SSO login'),
      '#states' => [
        'disabled' => [
          [':input[name="mod_sett_is_enabled"]' => ['value' => '0']],
        ],
      ]
    ];
  }

  private function buildDomainRestriction(
    array              &$form,
    FormStateInterface $form_state,
    bool               $disabled,
    MoModuleSettings   $entity,
    string             $current_lic_ver
  ): void
  {
    $form['domain_restriction_frame'] = [
      '#type' => 'details',
      '#tier' => MoModuleConstant::MO_ENTERPRISE_VERSION_TIER,
      '#title' => t('Domain Restriction'),
      '#open' => TRUE,
      '#states' => [
        'disabled' => [
          [':input[name="mod_sett_is_enabled"]' => ['value' => '0']],
        ],
      ]
    ];
    $form['domain_restriction_frame']['domain_restriction'] = [
      '#type' => 'checkbox',
      '#tier' => MoModuleConstant::MO_ENTERPRISE_VERSION_TIER,
      '#title' => t("Enable Domain Restriction"),
      '#default_value' => $entity->getDomainRestriction(),
      '#description' => t("Enabling this will restrict Users from particular email domains to access your site."),
      '#states' => [
        'disabled' => [
          [':input[name="mod_sett_is_enabled"]' => ['value' => '0']],
        ],
      ]
    ];
    $form['domain_restriction_frame']['restriction_frame'] = [
      '#type' => 'fieldset',
      '#states' => [
        'disabled' => [
          [':input[name="mod_sett_is_enabled"]' => ['value' => '0']],
        ],
        'visible' => [
          ':input[name="domain_restriction"]' => ['checked' => TRUE],
        ],
      ],
    ];
    $form['domain_restriction_frame']['restriction_frame']['allow_or_block_domains'] = array(
      '#type' => 'radios',
      '#tier' => MoModuleConstant::MO_ENTERPRISE_VERSION_TIER,
      '#options' => [
        'allow' => 'I want to allow only some of the domains',
        'block' => 'I want to block some of the domains'
      ],
      '#default_value' => $entity->getAllowOrBlockDomains() ?? 'allow',
      '#states' => [
        'disabled' => [
          [':input[name="mod_sett_is_enabled"]' => ['value' => '0']],
        ],
      ]
    );

    $restriction_page_domains = $entity->getRestrictedDomains();
    $domain_restriction_fields = self::domainRestrictionFields();
    $domain_restriction_header = self::domainRestrictionHeader();

    $custom_domain_rest_unique_id_array = MoAjaxTables::getUniqueID($form_state->get('domain_restriction_table_uid'), $restriction_page_domains);
    $form_state->set('domain_restriction_table_uid', $custom_domain_rest_unique_id_array);
    $form['domain_restriction_frame']['restriction_frame']['domain_restriction_table_frame'] = [
      '#type' => 'details',
      '#title' => $this->t('Domains'),
      '#open' => true,
      '#states' => [
        'disabled' => [
          [':input[name="mod_sett_is_enabled"]' => ['value' => '0']],
        ],
      ],
      'domain_restriction_table' => [
        'restricted_domains' => MoAjaxTables::generateTables(
          'domain-restriction-table',
          $domain_restriction_fields,
          $custom_domain_rest_unique_id_array,
          $restriction_page_domains,
          $domain_restriction_header
        ),
        'add_row' => MoAjaxTables::generateAddButton(
          'Add',
          [$this, 'addDomainRestrictionRowNew'],
          [$this, 'ajaxDomainCallback'],
          'domain-restriction-table',
          '',
          $disabled || !in_array($current_lic_ver, MoModuleConstant::MO_ENTERPRISE_VERSION_TIER)
        )]
    ];

  }

  private function buildSiteProtection(
    array              &$form,
    FormStateInterface $form_state,
    bool               $disabled,
    MoModuleSettings   $entity,
    string             $current_lic_ver
  ): void
  {
    $form['site_protection_frame'] = [
      '#type' => 'details',
      '#tier' => MoModuleConstant::MO_PREMIUM_VERSION_TIER,
      '#title' => t('Site Protections'),
      '#open' => TRUE,
    ];

    $client_configs = MoClientConfiguration::loadActive(false);

    $form['page_restriction_is_enabled'] = [
      '#type' => 'hidden',
      '#value' => (int)!($disabled || empty($client_configs))
    ];

    $config_page_url = Link::fromTextAndUrl(
      t('Client Configuration'),
      Url::fromRoute('entity.mo_client_config.add_form')
    )->toString();

    $form['site_protection_frame']['page_restriction_error_markup'] = [
      '#type' => 'markup',
      '#markup' => '<div class="mo-register-message">' .
        t(
          "Please configure at least 1&nbsp;application from the %config_tab&nbsp;tab to enable these&nbsp;features.",
          ['%config_tab' => $config_page_url]
        ) .
        '</div>',
      '#access' => empty($client_configs),
    ];
    /**@var MoClientConfiguration $config */

    $form['site_protection_frame']['page_restriction_app'] = [
      '#title' => t('Select a default Application for redirections'),
      '#tier' => MoModuleConstant::MO_PREMIUM_VERSION_TIER,
      '#id' => 'page_restriction_app',
      '#type' => 'select',
      '#default_value' => $entity->getPageRestrictionApp(),
      '#options' => array_map(fn($config) => $config->getAppName(), $client_configs),
      '#states' => [
        'visible' => [
          [':input[name = "replace_login_page"]' => ['checked' => TRUE]],
          [':input[name = "force_auth"]' => ['checked' => TRUE]],
        ],
        'disabled' => [
          [':input[name = "page_restriction_is_enabled"]' => ['value' => '0']],
        ]
      ]
    ];

    $form['site_protection_frame']['force_auth'] = [
      '#type' => 'checkbox',
      '#tier' => MoModuleConstant::MO_PREMIUM_VERSION_TIER,
      '#id' => 'force_auth',
      '#title' => t('Protect website against anonymous access'),
      '#default_value' => $entity->getForceAuth(),
      '#description' => t('Users will be redirected to your OAuth server for login in case user is not logged in and tries to access website.'),
      '#states' => [
        'disabled' => [
          [':input[name = "page_restriction_is_enabled"]' => ['value' => '0']],
        ]
      ],
    ];

    $form['site_protection_frame']['page_restriction_frame'] = [
      '#type' => 'details',
      '#tier' => MoModuleConstant::MO_ENTERPRISE_VERSION_TIER,
      '#title' => t('Page Restriction'),
      '#open' => in_array($current_lic_ver, MoModuleConstant::MO_ENTERPRISE_VERSION_TIER),
      '#states' => [
        'visible' => [
          [':input[name="force_auth"]' => ['checked' => TRUE]],
        ],
        'disabled' => [
          [':input[name = "page_restriction_is_enabled"]' => ['value' => '0']],
        ]
      ],
    ];

    $types_of_page_restrictions = [
      'whitelist_pages' => 'Pages to exclude from restriction',
      'restrict_pages' => 'Pages to be restricted'
    ];

    $form['site_protection_frame']['page_restriction_frame']['page_restriction_type'] = [
      '#type' => 'radios',
      '#tier' => MoModuleConstant::MO_ENTERPRISE_VERSION_TIER,
      '#options' => $types_of_page_restrictions,
      '#default_value' => $entity->getPageRestrictionType() ?? 'whitelist_pages',
    ];

    $form['site_protection_frame']['page_restriction_frame']['page_restriction_notes'] = [
      '#type' => 'item',
      '#title' => $this->t('Page Restriction Instructions'),
      '#markup' => $this->t(
        '<p>Enter the
                           <strong>relative URLs</strong>
                           . For example, if the site URL is
                           <strong>https://www.xyz.com/yyy</strong>
                           , the relative URL would be
                           <strong>/yyy</strong>.
                        </p>
                        <ul>
                            <li>To restrict or allow access to a specific route like
                                <strong>/abc/pqr/xyz</strong>
                                , enter that exact path.
                            </li>
                            <li>You can use the
                                <strong>*</strong>
                                wildcard to match multiple paths. For example,
                                <strong>/abc/*</strong>
                                will apply to all routes under
                                <strong>/abc</strong>.
                            </li>
                        </ul>'
      ),
    ];

    $restriction_page_urls = $entity->getRestrictedUrls();
    $page_restriction_url_fields = self::pageRestrictionUrlFields();
    $page_restriction_url_header = self::pageRestrictionUrlHeader();

    $custom_restriction_url_unique_id_array = MoAjaxTables::getUniqueID($form_state->get('page_restriction_url_table_uid'), $restriction_page_urls);
    $form_state->set('page_restriction_url_table_uid', $custom_restriction_url_unique_id_array);
    $form['site_protection_frame']['page_restriction_frame']['page_restriction_url_table_frame'] = [
      '#type' => 'details',
      '#tier' => MoModuleConstant::MO_ENTERPRISE_VERSION_TIER,
      '#title' => $this->t('Page Restriction URL\'s'),
      '#open' => true,
      'page_restriction_table' => [
        'restricted_urls' => MoAjaxTables::generateTables(
          'page-restriction-url-table',
          $page_restriction_url_fields,
          $custom_restriction_url_unique_id_array,
          $restriction_page_urls,
          $page_restriction_url_header,
        ),
        'add_row' => MoAjaxTables::generateAddButton(
          'Add',
          [$this, 'addPageRestrictionRow'],
          [$this, 'ajaxPageRestrictionCallback'],
          'page-restriction-url-table',
          '',
          $disabled || empty($client_configs) || !in_array($current_lic_ver, MoModuleConstant::MO_ENTERPRISE_VERSION_TIER)
        )]
    ];
    // NOTE: using same disabled (both in states as well directly in form) creates a complexity were the param in #state
    //  will get the priority and will get executed so added some small workaround by validating those operations in the
    //  #states part as well.
    $form['site_protection_frame']['replace_login_page'] = [
      '#type' => 'checkbox',
      '#tier' => MoModuleConstant::MO_PREMIUM_VERSION_TIER,
      '#id' => 'replace_login_page',
      '#title' => t('Replace Drupal login form with Identity Provider (OAuth Server) Login'),
      '#default_value' => $entity->getReplaceLoginPage(),
      '#description' => t('Users will be redirected to the configured Identity Provider (OAuth Server) for login when the Drupal login page (<i>' . Url::fromRoute('user.login', [], ['absolute' => true])->toString() . '</i>) is accessed.'),
      '#states' => [
        'disabled' => [
          [':input[name = "page_restriction_is_enabled"]' => ['value' => '0']],
        ]
      ]
    ];

    $form['site_protection_frame']['enable_page_access'] = [
      '#type' => 'checkbox',
      '#tier' => MoModuleConstant::MO_ENTERPRISE_VERSION_TIER,
      '#title' => t('Enable page access'),
      '#default_value' => $entity->getEnablePageAccess(),
      '#description' => t('Checking this option will enable the user to visit any page despite the redirection to OAuth Server settings (above two) are enabled.'),
      '#states' => [
        'visible' => [
          [':input[name="force_auth"]' => ['checked' => TRUE]],
          [':input[name="replace_login_page"]' => ['checked' => TRUE]],
        ],
        'disabled' => [
          [':input[name = "page_restriction_is_enabled"]' => ['value' => '0']],
        ]
      ],
    ];
    $form['site_protection_frame']['page_access_param'] = [
      '#type' => 'container',
      '#tier' => MoModuleConstant::MO_ENTERPRISE_VERSION_TIER,
      '#attributes' => ['class' => ['mo-url-container']],
      '#states' => [
        'visible' => [':input[name="enable_page_access"]' => ['checked' => TRUE]],
      ],
      'markup' => [
        '#type' => 'item',
        '#tier' => MoModuleConstant::MO_ENTERPRISE_VERSION_TIER,
        '#title' => t('Page Access Parameter:'),
        '#markup' => Markup::create(
          'Drupal page url followed by "' . '<i><b>' . '?mo_stop_redirect= true' . '"</i></b>',
        ),
      ]
    ];

    $form['site_protection_frame']['enable_backdoor'] = array(
      '#type' => 'checkbox',
      '#tier' => MoModuleConstant::MO_PREMIUM_VERSION_TIER,
      '#title' => t('Enable backdoor login'),
      '#default_value' => $entity->getEnableBackdoor(),
      '#description' => t(' Checking this option <b>creates a backdoor to login to your Website using Drupal credentials</b> in case you get locked out of your OAuth server.'),
      '#states' => [
        'visible' => [
          [':input[name="force_auth"]' => ['checked' => TRUE]],
          [':input[name="replace_login_page"]' => ['checked' => TRUE]],
        ],
        'disabled' => [
          [':input[name = "page_restriction_is_enabled"]' => ['value' => '0']],
        ]
      ]
    );


    $form['site_protection_frame']['backdoor_url'] = [
      '#type' => 'container',
      '#tier' => MoModuleConstant::MO_PREMIUM_VERSION_TIER,
      '#attributes' => ['class' => ['mo-url-container']],
      '#states' => [
        'visible' => [':input[name="enable_backdoor"]' => ['checked' => TRUE]],
        'disabled' => [
          [':input[name = "page_restriction_is_enabled"]' => ['value' => '0']],
        ]
      ],
      'markup' => [
        '#type' => 'item',
        '#tier' => MoModuleConstant::MO_PREMIUM_VERSION_TIER,
        '#title' => $this->t('Backdoor URL (Please note down following URL) :'),
        '#markup' => Markup::create(
          Url::fromRoute('user.login', ['oauth_client_login' => 'false'], ['absolute' => true])->toString(),
        ),
      ]
    ];

    // this piece is added in the ending because we don't want to keep the checkbox checked all the times even after update
    // issue explanation for not adding it to the parent form field:
    //   This makes the fields checked even if it is manually disabled by the user but the configuration stored in the backend aligns
    //      with the user saved config.
    if (!$entity->getForceAuth() && !$entity->getReplaceLoginPage()) {
      $form['site_protection_frame']['enable_page_access']['#states']['checked'] = [
        [':input[name="replace_login_page"]' => ['checked' => true]],
        [':input[name="force_auth"]' => ['checked' => true]],
      ];
      $form['site_protection_frame']['enable_backdoor']['#states']['checked'] = [
        [':input[name="replace_login_page"]' => ['checked' => true]],
        [':input[name="force_auth"]' => ['checked' => true]],
      ];
    }
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void
  {
    // @todo Validate the form here.
    parent::validateForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void
  {
    //todo alternative approach but it renders some additional fields which is not even relevant
    MoModuleSettings::save($form_state->cleanValues()->getValues());
    \Drupal::messenger()->addStatus("Module settings updated successfully.");
  }

  public function pageRestrictionUrlFields(): array
  {
    return [
      'page_url' => [
        'type' => 'textfield',
        'attributes' => ['placeholder' => $this->t('Page URL')],
      ],
      'delete_button_role' => [
        'type' => 'submit',
        'submit' => [$this, 'removePageRestrictionRow'],
        'callback' => [$this, 'ajaxPageRestrictionCallback'],
        'wrapper' => 'page-restriction-url-table',
      ],
    ];
  }

  public function tokenStorageTableFields(): array
  {
    return [
      'is_include' => [
        '#type' => 'checkbox',
      ],
      'token_type' => [
        '#type' => 'item',
      ],
      'token_key' => [
        '#type' => 'textfield',
//        //todo the below is one approach, it should always be paired with server side validation
//        'attributes' => [
//          'pattern' => "^[^=\-,; \t\r\n\013\014]+$",
//          'title' => 'Key Name cannot contain "=", ",", ";", " ", "\t", "\r", "\n", "\013", or "\014" characters',
//        ]
      ],
      'secure' => [
        '#type' => 'checkbox',
        '#states' => [
          'visible' => [
            ':input[name="store_token_in_cookies"]' => ['checked' => TRUE],
          ],
        ]
      ],
      'http_only' => [
        '#type' => 'checkbox',
        '#states' => [
          'visible' => [
            ':input[name="store_token_in_cookies"]' => ['checked' => TRUE],
          ],
        ]
      ],
    ];
  }

  public function domainRestrictionFields(): array
  {
    return [
      'domain' => [
        'type' => 'textfield',
        'attributes' => ['placeholder' => $this->t('Email Domain ex: @example.com')],
      ],
      'delete_button_role' => [
        'type' => 'submit',
        'submit' => [$this, 'removeDomainRow'],
        'callback' => [$this, 'ajaxDomainCallback'],
        'wrapper' => 'domain-restriction-table',
      ],
    ];
  }

  public function pageRestrictionUrlHeader(): array
  {
    return [
      ['data' => t('Page URL')],
      ['data' => t('Action')],
    ];
  }

  public function domainRestrictionHeader(): array
  {
    return [
      ['data' => t('Email Domain')],
      ['data' => t('Action')],
    ];
  }

  public function tokenStorageTableHeader(): array
  {
    return [
      ['data' => t('Include?')],
      ['data' => t('Token Type')],
      ['data' => t('Token Key Name')],
      ['data' => t('Secure')],
      ['data' => t('HttpOnly')],
    ];
  }

  public function addPageRestrictionRow(array &$form, FormStateInterface $form_state): void
  {
    $add_button = ($form_state->getTriggeringElement()['#name']);
    $rows = $form_state->getValue('total_rows_' . $add_button);
    $unique_array_id = $form_state->get('page_restriction_url_table_uid');
    for ($count = 1; $count <= $rows; $count++) {
      $uuid_service = \Drupal::service('uuid');
      $uuid = $uuid_service->generate();
      $unique_array_id[] = $uuid;
    }
    $form_state->set('page_restriction_url_table_uid', $unique_array_id);
    $form_state->setRebuild();
  }

  public function addDomainRestrictionRowNew(array &$form, FormStateInterface $form_state): void
  {
    $add_button = ($form_state->getTriggeringElement()['#name']);
    $rows = $form_state->getValue('total_rows_' . $add_button);
    $unique_array_id = $form_state->get('domain_restriction_table_uid');
    for ($count = 1; $count <= $rows; $count++) {
      $uuid_service = \Drupal::service('uuid');
      $uuid = $uuid_service->generate();
      $unique_array_id[] = $uuid;
    }
    $form_state->set('domain_restriction_table_uid', $unique_array_id);
    $form_state->setRebuild();
  }

  public function ajaxPageRestrictionCallback(array &$form, FormStateInterface $form_state)
  {
    return $form['mo_outer_container']['mo_outline']['site_protection_frame']['page_restriction_frame']['page_restriction_url_table_frame']['page_restriction_table']['restricted_urls'];
  }

  public function ajaxDomainCallback(array &$form, FormStateInterface $form_state)
  {
    return $form['mo_outer_container']['mo_outline']['domain_restriction_frame']['restriction_frame']['domain_restriction_table_frame']['domain_restriction_table']['restricted_domains'];
  }

  public function removePageRestrictionRow(array &$form, FormStateInterface $form_state): void
  {
    $id = $form_state->getTriggeringElement()['#name'];
    $var_value = 'page_restriction_url_table_uid';
    $unique_id = $form_state->get($var_value);
    $unique_id = array_diff($unique_id, [$id]);
    if (empty($unique_id)) {
      $uuid_service = \Drupal::service('uuid');
      $unique_id[0] = $uuid_service->generate();
    }
    $form_state->set($var_value, $unique_id);
    $form_state->setRebuild();
  }

  public function removeDomainRow(array &$form, FormStateInterface $form_state): void
  {
    $id = $form_state->getTriggeringElement()['#name'];
    $var_value = 'domain_restriction_table_uid';
    $unique_id = $form_state->get($var_value);
    $unique_id = array_diff($unique_id, [$id]);
    if (empty($unique_id)) {
      $uuid_service = \Drupal::service('uuid');
      $unique_id[0] = $uuid_service->generate();
    }
    $form_state->set($var_value, $unique_id);
    $form_state->setRebuild();
  }
}
