<?php

declare(strict_types=1);

namespace Drupal\miniorange_oauth_client\Form;

use Drupal\Core\Entity\EntityForm;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Link;
use Drupal\Core\Url;
use Drupal\miniorange_oauth_client\Entity\MoProfileMapping;
use Drupal\miniorange_oauth_client\MoConstant\MoModuleConstant;
use Drupal\miniorange_oauth_client\MoFeatures\MoUnoFeatures\MoLicense\MoLibrary\MoUnoSubscriptionLibrary;
use Drupal\miniorange_oauth_client\MoFeatures\MoUnoFeatures\MoUnoTestConfigAttributeListBuilder;
use Drupal\miniorange_oauth_client\MoFeatures\MoUnoFeatures\MoUnoTutorial;
use Drupal\miniorange_oauth_client\MoLibrary\MoGhostInvoker;
use Drupal\miniorange_oauth_client\MoLibrary\MoLogger;
use Drupal\miniorange_oauth_client\MoService\MoAjaxTables;
use Drupal\miniorange_oauth_client\MoService\MoLayoutBuilder;

/**
 * MiniOrange Profile Mapping Entity form.
 */
final class MoProfileMappingForm extends EntityForm {

  use MoLayoutBuilder;
  use MoGhostInvoker;
  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {

    /** @var MoUnoSubscriptionLibrary $sub_lib */
    $sub_lib = \Drupal::service('miniorange_oauth_client.uno_module_subscription_library');
    $disabled = $sub_lib->getModuleStatus($form);

    $profile_module_enabled = \Drupal::moduleHandler()->moduleExists('profile');
    if (!$profile_module_enabled) {
      $profile_module_url = Url::fromUri('https://www.drupal.org/project/profile', ['attributes' => ['target' => '_blank']]);
      $profile_link = Link::fromTextAndUrl('Profile module', $profile_module_url)->toString();
      $form['profile_mapping_not_available'] = [
        '#type' => 'markup',
        '#prefix' => '<div class="messages messages--warning">',
        '#suffix' => '</div>',
        '#markup' => $this->t('This extends the functionality of @profile_module', ['@profile_module' => $profile_link]),
      ];
    }
    /** @var MoProfileMapping $entity */
    $form = parent::buildForm($form, $form_state);
    $entity = $this->getEntity();

    $form['entity_key'] = [
      '#type' => 'hidden',
      '#value' => $entity->id(),
    ];

    $form['pro_map_is_enabled'] = [
      '#type' => 'hidden',
      '#value' => (int)(!$disabled && $profile_module_enabled),
    ];

    $form['enable_profile_mapping'] = [
      '#type' => 'checkbox',
      '#title' => t('Enable Profile Mapping'),
      '#tier' => MoModuleConstant::MO_ENTERPRISE_VERSION_TIER,
      '#default_value' => $entity->getEnableProfileMapping(),
      '#description' => t('Enable this first if you want to use any of the feature below.'),
      '#states' => [
        'disabled' => [
          [':input[name="pro_map_is_enabled"]' => ['value' => '0']],
        ],
      ]
    ];

    $form['profile_entity_type'] = [
      '#type' => 'select',
      '#tier' => MoModuleConstant::MO_ENTERPRISE_VERSION_TIER,
      '#title' => $this->t('Profile Type'),
      '#default_value' => $entity->getProfileEntityType(),
      '#options' => self::fetchProfileType(),
      '#states' => [
        'disabled' => [
          [':input[name="enable_profile_mapping"]' => ['checked' => FALSE]],
        ],
      ],
    ];

    $form['save_profile_entity_type'] = [
      '#type' => 'submit',
      '#tier' => MoModuleConstant::MO_ENTERPRISE_VERSION_TIER,
      '#button_type' => 'primary',
      '#value' => t('Save'),
      '#submit' => ['::submitForm', '::save'],
      '#states' => [
        'disabled' => [
          [':input[name="enable_profile_mapping"]' => ['checked' => FALSE]],
        ],
      ],
    ];

    $profile_mapping = $entity->getMappedFields();

    $profile_mapping_fields = $this->profileMappingFields();
    $profile_fields = $this->getProfileFields($entity->getProfileEntityType());
    $server_attrs = $profile_fields;

    $server_names_array = [1 => $server_attrs, 2 => $profile_mapping];
    $profile_mapping_header = self::ProfileMappingHeader();
    $profile_unique_id_array = MoAjaxTables::getUniqueID($form_state->get('profile_attribute_mapping_table_uid'), $profile_mapping);
    $form_state->set('profile_attribute_mapping_table_uid', $profile_unique_id_array);


    $form['profile_attr_map_frame'] = [
      '#type' => 'details',
      '#title' => t('Profile Attribute Mapping'),
      '#tier' => MoModuleConstant::MO_ENTERPRISE_VERSION_TIER,
      '#open' => TRUE,
      'mapping' => [
        'mapped_fields' => MoAjaxTables::generateTables(
          'profile-attribute-mapping-table',
          $profile_mapping_fields,
          $profile_unique_id_array,
          $profile_mapping,
          $profile_mapping_header,
          $server_names_array
        ),
        'add_row' => MoAjaxTables::generateAddButton(
          'Add',
          [$this, 'addRowNew'],
          [$this, 'ajaxCallback'],
          'profile-attribute-mapping-table',
          '',
        )
      ],
      '#states' => [
        'disabled' => [
          [':input[name="enable_profile_mapping"]' => ['checked' => FALSE]],
        ],
      ],
    ];
    if (!$entity->isNew()){
      $form['actions']['delete'] = [
        '#type' => 'submit',
        '#value' => $this->t('Reset'),
        "#weight" => 2,
        '#states' => [
          'disabled' => [
            [':input[name="enable_profile_mapping"]' => ['checked' => FALSE]],
          ],
        ],
        '#attributes' => ['class' => ['button', 'button--danger']],
        '#submit' => ['::delete']
      ];
    }

    $form['actions']['submit']['#attributes']['class'] = ['button', 'button--primary'];
    $form['actions']['submit']['#weight'] = 1;
    $form['actions']['submit']['#states'] = [
      'disabled' => [
        [':input[name="enable_profile_mapping"]' => ['checked' => FALSE]],
      ],
    ];


    $result['mo_outer_container'] = [
      "#type" => "container",
      "#attributes" => ['class' => ['mo-outer-container']]
    ];
    $this->modifyFormTitleOnTier($form);
    $result['mo_outer_container'] += $this->buildMOLayout($form);

    $result['mo_outer_container']['tutorial_frame'] = self::getTutorialLayout();

    $this->call(
      [MoUnoTutorial::class, 'buildVideoTutorialFrame'],
      ['result' => &$result['mo_outer_container']['tutorial_frame']]
    );
    $this->call(
      [MoUnoTestConfigAttributeListBuilder::class, 'buildTestConfigurationFrame'],
      ['result' => &$result['mo_outer_container']['tutorial_frame'], 'entity_id' => $entity->id()]
    );

    return $result;
  }

  public function submitForm(&$form, FormStateInterface $form_state): void
  {
    $form_state->setRedirect(
      'mo_configuration.edit', ['mo_client_config' => $this->entity->id(), 'tab' => 'profile_mapping']
    );
    parent::submitForm($form, $form_state);
  }

  public static function fetchProfileType(){
    $profile_types = ['' => '-- Select --'];

    if ( !\Drupal::service('module_handler')->moduleExists('profile') ){
      return $profile_types;
    }
    $types = \Drupal\profile\Entity\ProfileType::loadMultiple();

    foreach ($types as $type){
      $profile_types[$type->id()] = $type->label();
    }
    return $profile_types;
  }

  public function delete(array $form, FormStateInterface $form_state): void
  {
    try {
      $this->entity->delete();
    } catch (\Exception $exception) {
      //todo need to add catch block
    }
  }
  public function profileMappingFields(): array {
    return [
      'drupal_attribute_name' => [
        'type' => 'select',
      ],
      'oauth_attribute_name' => [
        'type' => 'textfield',
        'placeholder' => 'Semicolon (;) seperated values',
      ],
      'delete_button_role' => [
        'type' => 'submit',
        'submit' => [$this, 'removeRow'],
        'callback' => [$this, 'ajaxCallback'],
        'wrapper' => 'profile-attribute-mapping-table',
      ],
    ];
  }


  public function addRowNew(array &$form, FormStateInterface $form_state)
  {
    $add_button = ($form_state->getTriggeringElement()['#name']);
    $rows = $form_state->getValue('total_rows_' . $add_button);
    $unique_array_id = $form_state->get('profile_attribute_mapping_table_uid');
    for ($count = 1; $count <= $rows; $count++) {
      $uuid_service = \Drupal::service('uuid');
      $uuid = $uuid_service->generate();
      $unique_array_id[] = $uuid;
    }
    $form_state->set('profile_attribute_mapping_table_uid', $unique_array_id);
    $form_state->setRebuild();
  }
  public function ajaxCallback(array &$form, FormStateInterface $form_state) {
    return $form['mo_outer_container']['mo_outline']['profile_attr_map_frame']['mapping']['mapped_fields'];
  }

  public function removeRow(array &$form, FormStateInterface $form_state)
  {
    $id = $form_state->getTriggeringElement()['#name'];
    $var_value = 'profile_attribute_mapping_table_uid';
    $unique_id = $form_state->get($var_value);
    $unique_id = array_diff($unique_id, [$id]);
    if (empty($unique_id)) {
      $uuid_service = \Drupal::service('uuid');
      $unique_id[0] = $uuid_service->generate();
    }
    $form_state->set($var_value, $unique_id);
    $form_state->setRebuild();
  }

  private function getProfileFields(string $profile_type_id): array {
    $custom_fields = ['' => '-- Select --'];

    // Get current config entity for profile mapping (optional fallback)
    $entity = $this->getEntity();
    $enable_profile_mapping = $entity->getEnableProfileMapping();

    if (!$enable_profile_mapping) {
      return $custom_fields;
    }

    if (!\Drupal::service('module_handler')->moduleExists('profile')) {
      return $custom_fields;
    }

    if (empty($profile_type_id)) {
      return $custom_fields;
    }

    $uid = \Drupal::currentUser()->id();
    $user = \Drupal\user\Entity\User::load($uid);
    if (!$user) {
      return $custom_fields;
    }

    $user_profile_data = $user->toArray();
    $profile_key = $profile_type_id . '_profiles';

    try {
      if (isset($user_profile_data[$profile_key][0]['target_id'])) {
        $profile_id = $user_profile_data[$profile_key][0]['target_id'];
        $profile = \Drupal::entityTypeManager()
          ->getStorage('profile')
          ->load($profile_id)
          ->toArray();
      }
      // todo optional need to confirm the use case
      else {
        // Create dummy profile (optional fallback, useful for testing)
        $profile_entity = \Drupal\profile\Entity\Profile::create([
          'type' => $profile_type_id,
          'uid' => $uid,
        ]);
        $profile_entity->save();
        $profile = $profile_entity->toArray();
      }

      foreach ($profile as $key => $value) {
        if (strpos($key, 'field_') === 0) {
          $custom_fields[$key] = $key;
        }
      }

    } catch (\Exception $e) {
      MoLogger::error('Error accessing profile fields: @message', ['@message' => $e->getMessage()]);
    }

    return $custom_fields;
  }


  public function ProfileMappingHeader()
  {
    return [
      ['data' => t('Drupal Profile Field')],
      ['data' => t('OAuth Server Attribute Name')],
      ['data' => t('Action')],
    ];
  }

}
