<?php

namespace Drupal\miniorange_oauth_client\MoDTO;

use Drupal\miniorange_oauth_client\MoConstant\MoModuleConstant;
use Drupal\miniorange_oauth_client\MoLibrary\MoCacheLibrary;
use ReflectionClass;
use ReflectionProperty;

final class MoModuleSettings
{
  protected ?bool $enforce_https_callback = false;
  protected ?string $page_restriction_app = null;
  protected ?bool $force_auth = false;
  protected ?string $page_restriction_type = null;
  protected ?array $restricted_urls = [];
  protected ?bool $replace_login_page = false;
  protected ?bool $enable_page_access = false;
  protected ?bool $enable_backdoor = false;
  protected ?bool $domain_restriction = false;
  protected ?string $allow_or_block_domains = null;
  protected ?array $restricted_domains = [];
  protected ?bool $make_profile_fields_required = false;
  protected ?string $user_profile_required_attributes = null;
  protected ?bool $make_profile_field_disable = false;
  protected ?string $user_profile_disable_attributes = null;
  protected ?string $password_and_confirm_password_status = 'editable';
  protected ?bool $duplicate_username = false;
  protected ?bool $change_status_to_active_on_sso = false;
  protected ?bool $store_token_in_session = false;
  protected ?bool $store_token_in_cookies = false;
  protected ?array $token_storage_keys = [];

  /**
   * @param bool|null $enforce_https_callback
   */
  public function __construct(?array $config = [])
  {
    foreach ($config as $key => $value) {
      $this->$key = $value ?? $this->$key;
    }
  }

  public static function buildArray(?array $getValues = []): array
  {
    $reflect = new ReflectionClass(self::class);
    $props = $reflect->getProperties(
      ReflectionProperty::IS_PRIVATE | ReflectionProperty::IS_PROTECTED | ReflectionProperty::IS_PUBLIC
    );
    foreach ($props as $prop) {
        $array[$prop->getName()] = $getValues[$prop->getName()];
    }
    return $array ?? [];
  }

  public function toArray()
  {
    $reflect = new ReflectionClass(self::class);
    $props = $reflect->getProperties(
      ReflectionProperty::IS_PRIVATE | ReflectionProperty::IS_PROTECTED | ReflectionProperty::IS_PUBLIC
    );
    foreach ($props as $prop) {
      $array[$prop->getName()] = $this->{$prop->getName()};
    }
    return $array ?? [];
  }
  /**
   * @throws \Exception
   */
  public static function save(?array $form_values = []): void
  {
    $config = \Drupal::configFactory()->getEditable(MoModuleConstant::MODULE_SETTINGS);
    foreach (MoModuleSettings::buildArray($form_values) as $key => $value) {
      $config->set($key, $value);
    }
    $config->save();
    MoCacheLibrary::removeCache(MoModuleConstant::MODULE_SETTINGS);
  }

  public static function readMe(): self
  {
    return new self(MoCacheLibrary::readCache(MoModuleConstant::MODULE_SETTINGS));
  }

  /**
   * @throws \Exception
   */
  public static function remove(): void {
    $config = \Drupal::configFactory()->getEditable(MoModuleConstant::MODULE_SETTINGS);
    $config->delete();
    MoCacheLibrary::removeCache(MoModuleConstant::MODULE_SETTINGS);
    \Drupal::messenger()->addStatus("Module settings cleared successfully.");
  }

  public function getEnforceHttpsCallback(): ?bool
  {
    return $this->enforce_https_callback;
  }

  public function getPageRestrictionApp(): ?string
  {
    return $this->page_restriction_app;
  }

  public function getForceAuth(): ?bool
  {
    return $this->force_auth;
  }

  public function getPageRestrictionType(): ?string
  {
    return $this->page_restriction_type;
  }

  public function getRestrictedUrls(): ?array
  {
    return $this->restricted_urls;
  }

  public function getReplaceLoginPage(): ?bool
  {
    return $this->replace_login_page;
  }

  public function getEnablePageAccess(): ?bool
  {
    return $this->enable_page_access;
  }

  public function getEnableBackdoor(): ?bool
  {
    return $this->enable_backdoor;
  }

  public function getDomainRestriction(): ?bool
  {
    return $this->domain_restriction;
  }

  public function getAllowOrBlockDomains(): ?string
  {
    return $this->allow_or_block_domains;
  }

  public function getRestrictedDomains(): ?array
  {
    return $this->restricted_domains;
  }

  public function getMakeProfileFieldsRequired(): ?bool
  {
    return $this->make_profile_fields_required;
  }

  public function getUserProfileRequiredAttributes(): ?string
  {
    return $this->user_profile_required_attributes;
  }

  public function getMakeProfileFieldDisable(): ?bool
  {
    return $this->make_profile_field_disable;
  }

  public function getUserProfileDisableAttributes(): ?string
  {
    return $this->user_profile_disable_attributes;
  }

  public function getPasswordAndConfirmPasswordStatus(): ?string
  {
    return $this->password_and_confirm_password_status;
  }

  public function getDuplicateUsername(): ?bool
  {
    return $this->duplicate_username;
  }

  public function getChangeStatusToActiveOnSso(): ?bool
  {
    return $this->change_status_to_active_on_sso;
  }

  public function getStoreTokenInSession(): ?bool
  {
    return $this->store_token_in_session;
  }

  public function getStoreTokenInCookies(): ?bool
  {
    return $this->store_token_in_cookies;
  }

  public function getTokenStorageKeys(): ?array
  {
    return empty($this->token_storage_keys) ? $this->generateBasicTokenStorageRow() : $this->token_storage_keys;
  }


  private function generateBasicTokenStorageRow(): array
  {
    $rows = [];
    foreach (['Id Token', 'Access Token', 'Refresh Token'] as $token_type) {
      $rows[str_replace(' ', '_', strtolower($token_type))] = [
        'is_include' => 0,
        'token_type' => $token_type,
        'token_key' => str_replace(' ', '_', strtolower($token_type)),
        'secure' => 0,
        'http_only' => 0
      ];
    }
    return $rows;
  }
}
