<?php

namespace Drupal\miniorange_oauth_client\MoHelper\JWTHandler\Utils;

class JWSVerify {

	public $algo;

	public function __construct(string $alg = '') {
		if ( empty( $alg ) ) {
			return;
		}
		$alg = explode( 'S', $alg );
		if (!is_array($alg) || 2 !== count( $alg ) ) {
      throw new \Exception('The Signature for id_token seems to be invalid or unsupported.');
		}
		if ( 'H' === $alg[0] ) {
			$this->algo['alg'] = 'HSA';
		} elseif ( 'R' === $alg[0] ) {
			$this->algo['alg'] = 'RSA';
		} else {
      throw new \Exception('The Signature for id_token seems to be invalid or unsupported.');
		}
		$this->algo['sha'] = $alg[1];
	}

	private function validate_hmac( $payload_to_verify = '', $secret = '', $sign = '' ) {
		if ( empty( $payload_to_verify ) || empty( $sign ) ) {
			return false;
		}
		$sha      = $this->algo['sha'];
		$sha      = 'sha' . $sha;
		$new_sign = hash_hmac( $sha, $payload_to_verify, $secret, true );
		return hash_equals( $new_sign, $sign );
	}

	private function validate_rsa( $payload_to_verify = '', $raw_cert = '', $sign = '' ) {
		if ( empty( $payload_to_verify ) || empty( $sign ) ) {
			return false;
		}
		$sha        = $this->algo['sha'];
		$public_key = '';
		$parts      = explode( '-----', $raw_cert );
		if ( preg_match( '/\r\n|\r|\n/', $parts[2] ) ) {
			$public_key = $raw_cert;
		} else {
			$encoding = '-----' . $parts[1] . "-----\n";
			$offset   = 0;
			while ( $segment = substr( $parts[2], $offset, 64 ) ) {
				$encoding .= $segment . "\n";
				$offset   += 64;
			}
			$encoding  .= '-----' . $parts[3] . "-----\n";
			$public_key = $encoding;
		}
		$verified = false;
		switch ( $sha ) {
			case '256':
				$verified = openssl_verify( $payload_to_verify, $sign, $public_key, OPENSSL_ALGO_SHA256 );
				break;
			case '384':
				$verified = openssl_verify( $payload_to_verify, $sign, $public_key, OPENSSL_ALGO_SHA384 );
				break;
			case '512':
				$verified = openssl_verify( $payload_to_verify, $sign, $public_key, OPENSSL_ALGO_SHA512 );
				break;
			default:
				$verified = false;
				break;
		}
		return $verified;
	}

	public function verify( $payload_to_verify = '', $secret = '', $sign = '' ) {
		if ( empty( $payload_to_verify ) || empty( $sign ) ) {
			return false;
		}
		$alg = $this->algo['alg'];
		switch ( $alg ) {
			case 'HSA':
				return $this->validate_hmac( $payload_to_verify, $secret, $sign );
			case 'RSA':
				return @$this->validate_rsa( $payload_to_verify, $secret, $sign );
			default:
				return false;
		}
	}
}

